"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildStartBuild = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Start a CodeBuild Build as a task.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
 */
class CodeBuildStartBuild extends sfn.TaskStateBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskMetrics = {
            metricPrefixSingular: 'CodeBuildProject',
            metricPrefixPlural: 'CodeBuildProjects',
            metricDimensions: {
                ProjectArn: this.props.project.projectArn,
            },
        };
        this.taskPolicies = this.configurePolicyStatements();
    }
    configurePolicyStatements() {
        let policyStatements = [
            new iam.PolicyStatement({
                resources: [this.props.project.projectArn],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:BatchGetBuilds',
                    'codebuild:BatchGetReports',
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    cdk.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventForCodeBuildStartBuildRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Provides the CodeBuild StartBuild service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('codebuild', 'startBuild', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ProjectName: this.props.project.projectName,
                EnvironmentVariablesOverride: this.props.environmentVariablesOverride
                    ? this.serializeEnvVariables(this.props.environmentVariablesOverride)
                    : undefined,
            }),
        };
    }
    serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            Name: name,
            Type: environmentVariables[name].type || codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            Value: environmentVariables[name].value,
        }));
    }
}
exports.CodeBuildStartBuild = CodeBuildStartBuild;
CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhcnQtYnVpbGQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFydC1idWlsZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxvREFBb0Q7QUFDcEQsd0NBQXdDO0FBQ3hDLGtEQUFrRDtBQUNsRCxxQ0FBcUM7QUFFckMsc0RBQXlGOzs7Ozs7QUF1QnpGLE1BQWEsbUJBQW9CLFNBQVEsR0FBRyxDQUFDLGFBQWE7Ozs7SUFXeEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBbUIsS0FBK0I7O1FBQ3hGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRGlDLFVBQUssR0FBTCxLQUFLLENBQTBCO1FBRXhGLElBQUksQ0FBQyxrQkFBa0IsU0FBRyxLQUFLLENBQUMsa0JBQWtCLG1DQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsQ0FBQztRQUU5RixxQ0FBd0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsbUJBQW1CLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUV0RyxJQUFJLENBQUMsV0FBVyxHQUFHO1lBQ2pCLG9CQUFvQixFQUFFLGtCQUFrQjtZQUN4QyxrQkFBa0IsRUFBRSxtQkFBbUI7WUFDdkMsZ0JBQWdCLEVBQUU7Z0JBQ2hCLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVO2FBQzFDO1NBQ0YsQ0FBQztRQUVGLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7SUFDdkQsQ0FBQztJQUVPLHlCQUF5QjtRQUMvQixJQUFJLGdCQUFnQixHQUFHO1lBQ3JCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO2dCQUMxQyxPQUFPLEVBQUU7b0JBQ1Asc0JBQXNCO29CQUN0QixxQkFBcUI7b0JBQ3JCLDBCQUEwQjtvQkFDMUIsMkJBQTJCO2lCQUM1QjthQUNGLENBQUM7U0FDSCxDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEtBQUssR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU8sRUFBRTtZQUM5RCxnQkFBZ0IsQ0FBQyxJQUFJLENBQ25CLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCLENBQUM7Z0JBQ3ZFLFNBQVMsRUFBRTtvQkFDVCxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7d0JBQzNCLE9BQU8sRUFBRSxRQUFRO3dCQUNqQixRQUFRLEVBQUUsc0RBQXNEO3FCQUNqRSxDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUNILENBQUM7U0FDSDtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQztJQUVEOztPQUVHO0lBQ0g7O09BRUc7SUFDTyxXQUFXO1FBQ25CLE9BQU87WUFDTCxRQUFRLEVBQUUsbUNBQXNCLENBQUMsV0FBVyxFQUFFLFlBQVksRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUM7WUFDcEYsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDO2dCQUN0QyxXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDM0MsNEJBQTRCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyw0QkFBNEI7b0JBQ25FLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsQ0FBQztvQkFDckUsQ0FBQyxDQUFDLFNBQVM7YUFDZCxDQUFDO1NBQ0gsQ0FBQztJQUNKLENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxvQkFBNEU7UUFDeEcsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNwRCxJQUFJLEVBQUUsSUFBSTtZQUNWLElBQUksRUFBRSxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLElBQUksU0FBUyxDQUFDLDRCQUE0QixDQUFDLFNBQVM7WUFDekYsS0FBSyxFQUFFLG9CQUFvQixDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUs7U0FDeEMsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDOztBQWxGSCxrREFtRkM7QUFsRnlCLGtEQUE4QixHQUE2QjtJQUNqRixHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCO0lBQ3ZDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPO0NBQy9CLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlYnVpbGQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBpbnRlZ3JhdGlvblJlc291cmNlQXJuLCB2YWxpZGF0ZVBhdHRlcm5TdXBwb3J0ZWQgfSBmcm9tICcuLi9wcml2YXRlL3Rhc2stdXRpbHMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ29kZUJ1aWxkU3RhcnRCdWlsZFByb3BzIGV4dGVuZHMgc2ZuLlRhc2tTdGF0ZUJhc2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb2plY3Q6IGNvZGVidWlsZC5JUHJvamVjdDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnZpcm9ubWVudFZhcmlhYmxlc092ZXJyaWRlPzogeyBbbmFtZTogc3RyaW5nXTogY29kZWJ1aWxkLkJ1aWxkRW52aXJvbm1lbnRWYXJpYWJsZSB9O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIENvZGVCdWlsZFN0YXJ0QnVpbGQgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZSB7XG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUzogc2ZuLkludGVncmF0aW9uUGF0dGVybltdID0gW1xuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRSxcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gIF07XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tNZXRyaWNzPzogc2ZuLlRhc2tNZXRyaWNzQ29uZmlnO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza1BvbGljaWVzPzogaWFtLlBvbGljeVN0YXRlbWVudFtdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IENvZGVCdWlsZFN0YXJ0QnVpbGRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuID8/IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRTtcblxuICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgQ29kZUJ1aWxkU3RhcnRCdWlsZC5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuXG4gICAgdGhpcy50YXNrTWV0cmljcyA9IHtcbiAgICAgIG1ldHJpY1ByZWZpeFNpbmd1bGFyOiAnQ29kZUJ1aWxkUHJvamVjdCcsXG4gICAgICBtZXRyaWNQcmVmaXhQbHVyYWw6ICdDb2RlQnVpbGRQcm9qZWN0cycsXG4gICAgICBtZXRyaWNEaW1lbnNpb25zOiB7XG4gICAgICAgIFByb2plY3RBcm46IHRoaXMucHJvcHMucHJvamVjdC5wcm9qZWN0QXJuLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgdGhpcy50YXNrUG9saWNpZXMgPSB0aGlzLmNvbmZpZ3VyZVBvbGljeVN0YXRlbWVudHMoKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uZmlndXJlUG9saWN5U3RhdGVtZW50cygpOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgIGxldCBwb2xpY3lTdGF0ZW1lbnRzID0gW1xuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICByZXNvdXJjZXM6IFt0aGlzLnByb3BzLnByb2plY3QucHJvamVjdEFybl0sXG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAnY29kZWJ1aWxkOlN0YXJ0QnVpbGQnLFxuICAgICAgICAgICdjb2RlYnVpbGQ6U3RvcEJ1aWxkJyxcbiAgICAgICAgICAnY29kZWJ1aWxkOkJhdGNoR2V0QnVpbGRzJyxcbiAgICAgICAgICAnY29kZWJ1aWxkOkJhdGNoR2V0UmVwb3J0cycsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICBdO1xuXG4gICAgaWYgKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID09PSBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IpIHtcbiAgICAgIHBvbGljeVN0YXRlbWVudHMucHVzaChcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFsnZXZlbnRzOlB1dFRhcmdldHMnLCAnZXZlbnRzOlB1dFJ1bGUnLCAnZXZlbnRzOkRlc2NyaWJlUnVsZSddLFxuICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgIHNlcnZpY2U6ICdldmVudHMnLFxuICAgICAgICAgICAgICByZXNvdXJjZTogJ3J1bGUvU3RlcEZ1bmN0aW9uc0dldEV2ZW50Rm9yQ29kZUJ1aWxkU3RhcnRCdWlsZFJ1bGUnLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgIH1cblxuICAgIHJldHVybiBwb2xpY3lTdGF0ZW1lbnRzO1xuICB9XG5cbiAgLyoqXG4gICAqIFByb3ZpZGVzIHRoZSBDb2RlQnVpbGQgU3RhcnRCdWlsZCBzZXJ2aWNlIGludGVncmF0aW9uIHRhc2sgY29uZmlndXJhdGlvblxuICAgKi9cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHJvdGVjdGVkIF9yZW5kZXJUYXNrKCk6IGFueSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFJlc291cmNlOiBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdjb2RlYnVpbGQnLCAnc3RhcnRCdWlsZCcsIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuKSxcbiAgICAgIFBhcmFtZXRlcnM6IHNmbi5GaWVsZFV0aWxzLnJlbmRlck9iamVjdCh7XG4gICAgICAgIFByb2plY3ROYW1lOiB0aGlzLnByb3BzLnByb2plY3QucHJvamVjdE5hbWUsXG4gICAgICAgIEVudmlyb25tZW50VmFyaWFibGVzT3ZlcnJpZGU6IHRoaXMucHJvcHMuZW52aXJvbm1lbnRWYXJpYWJsZXNPdmVycmlkZVxuICAgICAgICAgID8gdGhpcy5zZXJpYWxpemVFbnZWYXJpYWJsZXModGhpcy5wcm9wcy5lbnZpcm9ubWVudFZhcmlhYmxlc092ZXJyaWRlKVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgfSksXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgc2VyaWFsaXplRW52VmFyaWFibGVzKGVudmlyb25tZW50VmFyaWFibGVzOiB7IFtuYW1lOiBzdHJpbmddOiBjb2RlYnVpbGQuQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlIH0pIHtcbiAgICByZXR1cm4gT2JqZWN0LmtleXMoZW52aXJvbm1lbnRWYXJpYWJsZXMpLm1hcChuYW1lID0+ICh7XG4gICAgICBOYW1lOiBuYW1lLFxuICAgICAgVHlwZTogZW52aXJvbm1lbnRWYXJpYWJsZXNbbmFtZV0udHlwZSB8fCBjb2RlYnVpbGQuQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlVHlwZS5QTEFJTlRFWFQsXG4gICAgICBWYWx1ZTogZW52aXJvbm1lbnRWYXJpYWJsZXNbbmFtZV0udmFsdWUsXG4gICAgfSkpO1xuICB9XG59XG4iXX0=