# cython: boundscheck=True, wraparound=True
import numpy as np

__all__ = ['tessellate', 'facet_size', 'find_largest_facets', 
           'group_facets_by_size', 'find_neighbors',
           'identify_facets_and_directions', 'identify_facets',
           'facets_adjacency_matrix', 'facet_coloring', 'facet_normals']

#------------------------------------------------------------------------------
def tessellate(int[::1] cables_idx, int[::1] cable_len, int cable_sep, 
               char[::1] sign, int return_moves=True, int return_neigh=False):
    """Create a list of facets that tesselate the surface based on the cables

    Args:
        cables_idx (int32 array): concatenation of arrays of vertex indices, 
            each descibing a cable (all longitudinal cables are listed first,
            then the transverse cables)
        cables_len (int32 array): length of each cable, such that 
            sum(cables_len) == cables.size
        cable_sep (int): number of longitudinal cables
        sign (uint8 array): generated by ceate_cable_network; it relates to 
            the surface orientation
        return_moves (bool): also return the path for each facet (see below); 
            default: True
        return_neigh (bool): also return the neighbors of all vertices (see 
            below); default: False
    
    Returns:
        facets (list of tuples or list of arrays): each facet is represented by
            an array of vertex indices and, if return_moves is True, by a 
            string indicating of the  moves are in the left (L), right (R) or 
            forward (F) direction, e.g. 'LLLL' means the path requires to go 
            4 times to the left to ceate a square
        vneigh (nv-by-4 int array): if return_neigh is True, returns the 
            indices of up to 4 neighboring vertices of each vertex; 
            the first two columns are for neighbors in the longitudinal 
            direction, the next two columns for the transverse direction; 
            index is -1 where there is no neighbor
    """
    vneigh, halfedges, heneigh = find_neighbors(cables_idx, cable_len, 
                                                cable_sep, sign)
    if return_moves:
        output = identify_facets_and_directions(halfedges, heneigh)
    else:
        output = identify_facets(halfedges, heneigh)
    if return_neigh:
        return output, vneigh
    return output

#------------------------------------------------------------------------------
def facet_size(list facets):
    """Calculate facet sizes

    Args:
        facets (list of int arrays or list of tuples): generated by 
            'tessellate'

    Returns:
        int array: number of vertices of each facet
    """
    if not facets:
        return np.empty(0, dtype=np.int32)
    if isinstance(facets[0], tuple):
        facet_size = [f.size for f, _ in facets]
    else:
        facet_size = [f.size for f in facets]
    return np.array(facet_size, dtype=np.int32)

#------------------------------------------------------------------------------
def find_largest_facets(list facets, int n):
    """Find the n lagest facets in a list of facets

    Args:
        facets (list of int arrays or list of tuples): generated by 
            'tessellate'
        n (int): number of facets to extract
    
    Returns:
        idx (int array of size n): indices of the n largest facets
        cutoff (int): size of the n-th largest facet
    """
    if len(facets) <= n:
        if not facets:
            return np.empty(0, dtype=np.int32), 0
        return (np.arange(len(facets), dtype=np.int32), 
                min(f.size for f in facets))
    sizes = facet_size(facets)
    idx =  np.argpartition(sizes, len(sizes)-n)[-n:]
    cutoff = sizes[idx].min()
    return idx, cutoff

#------------------------------------------------------------------------------
def group_facets_by_size(list facets, int cutoff=1000000):
    """Group polygonal facets by their number of sides

    Args:
        facets (list of int arrays or list of tuples): generated by 
            'tessellate'
        cutoff (int): facets with number of sides >= cutof are identified as
            boundaries
    
    Returns:
        facet_groups (list of arrays): facet_groups[n] is a k-by-n array whose 
            k rows are facets with n sides
        boundaries (list of arrays): facets with more than 'cutoff' sides are
            listed separately in this list
        mapping (int array): for each output facet, gives the index of the 
            facet in the original list
    """
    cdef:
        int n, i, k, max_size, is_tuple, i_facet
        list boundaries, facet_groups, facetid_groups
    
    if not facets:
        return [], [], np.empty(0, dtype=np.int32)
    sizes = facet_size(facets)
    sizes_below_cutoff = sizes[sizes<cutoff]
    if sizes_below_cutoff.size:
        max_size = sizes_below_cutoff.max()
    else:
        max_size = 0
    is_tuple = isinstance(facets[0], tuple)

    boundaries = []
    facet_groups = []
    facetid_groups = []
    for n in range(max_size+1):
        facet_groups.append([])
        facetid_groups.append([])
    
    i_facet = 0
    for i in range(len(facets)):
        if is_tuple:
            poly = facets[i][0]
        else:
            poly = facets[i]
        k = poly.size
        if k <= max_size:
            facet_groups[k].append(poly)
            facetid_groups[k].append(i_facet)
            i_facet += 1
        else:
            boundaries.append(poly)

    for n in range(max_size+1):
        if facet_groups[n]:
            facet_groups[n] = np.row_stack(facet_groups[n])
        else:
            facet_groups[n] = np.empty((0, n), dtype=np.int32)

    for n in range(max_size+1):
        facetid_groups[n] = np.array(facetid_groups[n], dtype=np.int32)
    
    mapping = np.concatenate(facetid_groups)
    return facet_groups, boundaries, mapping

#------------------------------------------------------------------------------
def find_neighbors(int[::1] cables_idx, int[::1] cable_len, int cable_sep, 
                   char[::1] sign):
    """Deterning the nearest neigbors of vertices, ceate the set of the edges 
    and find the neighboring edges of an edge

    Args:
        cables_idx (int array): concatenation of arrays of vertex indices, each 
            descibing a cable (all longitudinal cables are listed first, 
            then the transverse cables)
        cables_len (int array): length of each cable, such that 
            sum(cables_len) == cables.size
        cable_sep (int): number of longitudinal cables
        sign (uint8 array): generated by ceate_cable_network; it relates to 
            the surface orientation
    
    Returns:
        vneigh (nv-by-4 int array): indices of up to 4 neighboring vertices of 
            each vertex; the first two columns are for neighbors in the 
            longitudinal direction, the next two columns for the transverse 
            direction; index is -1 where there is no neighbor
        halfedges (2*ne-by-2 int array): indices of the vertices of all
            half-edges (for each edge, there are two half-edges whose vertex 
            indices are exchanged)
        heneigh (2*ne-by-3 int array): indices of up to 3 neighboring 
            half-edges of each half-edge; the three columns correspond to 
            half-edges in the left, forward and right directions respectively 
            (relative to the orientation of the triangular mesh); index is -1 
            if there is no neighbor
    """
    cdef:
        int nv, nc, c, clen, i, j=0, k=0, m, v1, v2, shift, nhe=0, e, b
        int[:, :] vneigh
        int[:, :] heneigh
        int[:, :] halfedges
        int[:, :] nexthalfedges
        int[:] halfedge_cable

    nc = cable_len.size
    nv = sign.size
    for i in range(nc):
        nhe += max(2 * cable_len[i] - 2, 0)
    vneigh_arr = np.full((nv, 4), -1, dtype=np.int32)
    heneigh_arr = np.full((nhe, 3), -1, dtype=np.int32)
    halfedges_arr = np.full((nhe, 2), -1, dtype=np.int32)
    vneigh = vneigh_arr
    heneigh = heneigh_arr
    halfedges = halfedges_arr
    nexthalfedges = np.full((nv, 4), -1, dtype=np.int32)
    halfedge_cable = np.full(nhe, -1, dtype=np.int32)

    for c in range(nc):
        shift = 0 if c < cable_sep else 2
        clen = cable_len[c]
        if clen == 0:
            continue
        for i in range(clen-1):
            v1 = cables_idx[j]
            v2 = cables_idx[j+1]

            vneigh[v1, 1+shift] = v2
            vneigh[v2, 0+shift] = v1

            halfedges[k, 0] = v2 # even k -> backward
            halfedges[k, 1] = v1
            halfedges[k+1, 0] = v1 # odd k -> forward
            halfedges[k+1, 1] = v2
            halfedge_cable[k] = halfedge_cable[k+1] = c

            for m in range(4):
                if nexthalfedges[v1, m] == -1:
                    nexthalfedges[v1, m] = k+1
                    break
            for m in range(4):
                if nexthalfedges[v2, m] == -1:
                    nexthalfedges[v2, m] = k
                    break

            if i > 0:
                heneigh[k, 1] = k-2 # forward
            if i < clen-2:
                heneigh[k+1, 1] = k+3 # forward

            k += 2
            j += 1
        j += 1

    for k in range(nhe):
        i = halfedges[k, 1]
        c = halfedge_cable[k]
        for m in range(4):
            e = nexthalfedges[i, m]
            if e < 0:
                break
            if halfedge_cable[e] == c:
                continue
            b = sign[i] == 1 # positive orientation
            if e % 2 == 0: # backward
                b = not b
            if k % 2 == 0: # backward
                b = not b
            if c < cable_sep:
                b = not b
            if not b:
                heneigh[k, 0] = e # left
            else:
                heneigh[k, 2] = e # right
    
    return vneigh_arr, halfedges_arr, heneigh_arr

#------------------------------------------------------------------------------
def identify_facets_and_directions(int[:, ::1] halfedges, int[:, ::1] heneigh):
    """Identify facets from the list of halfedges and their neighbors

    Args:
        halfedges (2*ne-by-2 int array): indices of the vertices of all
            half-edges (for each edge, there are two half-edges whose vertex 
            indices are exchanged)
        heneigh (2*ne-by-3 int array): indices of up to 3 neighboring 
            half-edges of each half-edge; the three columns correspond to 
            half-edges in the left, forward and right directions respectively 
            (relative to the orientation of the triangular mesh); index is -1 
            if there is no neighbor
    
    Returns:
        facets (list of tuples): each facet is represented by an array of 
            vertex indices and by a string indicating of the moves are in the
            left (L), right (R) or forward (F) direction, e.g. 'LLLL' means the
            path requires to go 4 times to the left to ceate a square
    """
    cdef:
        list loop, facets = []
        str moves
        int k0 = 0, k, m, e, nhe
        char [::1] taken
    nhe = halfedges.shape[0]
    taken = np.zeros(nhe, dtype=np.uint8)
    while True:
        while k0 < nhe and taken[k0]:
            k0 += 1
        if k0 >= nhe:
            break
        loop = [halfedges[k0, 1]]
        moves = ''
        k = k0
        taken[k0] = 1
        while len(loop) < nhe:
            for m in range(3):
                e = heneigh[k, m]
                if e >= 0:
                    break
            moves += 'LFR'[m]
            assert e >= 0
            taken[e] = 1
            if e == k0:
                break
            loop.append(halfedges[e, 1])
            k = e
        facets.append((np.array(loop, dtype=np.int32), moves))
    return facets

#------------------------------------------------------------------------------
def identify_facets(int[:, ::1] halfedges, int[:, ::1] heneigh):
    """Identify facets from the list of halfedges and their neighbors

    Args:
        halfedges (2*ne-by-2 int array): indices of the vertices of all
            half-edges (for each edge, there are two half-edges whose vertex 
            indices are exchanged)
        heneigh (2*ne-by-3 int array): indices of up to 3 neighboring 
            half-edges of each half-edge; the three columns correspond to 
            half-edges in the left, forward and right directions respectively 
            (relative to the orientation of the triangular mesh); index is -1 
            if there is no neighbor
    
    Returns:
        facets (list of int arrays): each facet is represented by an array of 
        vertex indices
    """
    cdef:
        list loop, facets = []
        str moves
        int k0 = 0, k, m, e, nhe
        char [::1] taken
    nhe = halfedges.shape[0]
    taken = np.zeros(nhe, dtype=np.uint8)
    while True:
        while k0 < nhe and taken[k0]:
            k0 += 1
        if k0 >= nhe:
            break
        loop = [halfedges[k0, 1]]
        k = k0
        taken[k0] = 1
        while len(loop) < nhe:
            for m in range(3):
                e = heneigh[k, m]
                if e >= 0:
                    break
            assert e >= 0
            taken[e] = 1
            if e == k0:
                break
            loop.append(halfedges[e, 1])
            k = e
        facets.append(np.array(loop, dtype=np.int32))
    return facets

#------------------------------------------------------------------------------
def facets_adjacency_matrix(list facets):
    """Adjacency matrix for the list of facets. Facets are adjacent if they 
    share an edge.

    Args:
        facets (list): list of numpy arrays of vertex indices forming the 
            facets; it may be necessary to eliminate the facet(s) corresponding
            to the boundaries
    
    Returns:
        list of sets: for each facet, the set of adjacent facets
    """
    cdef:
        list adj
        dict edges = {}
        int i, j, j1, is_tuple, a, b
        tuple pair

    if not facets:
        return []
    adj = [set() for i in range(len(facets))]
    is_tuple = isinstance(facets[0], tuple)

    for i, f in enumerate(facets):
        if is_tuple:
            f = f[0]
        for j in range(f.size):
            j1 = j+1
            if j1 == f.size:
                j1 = 0
            a = f[j]
            b = f[j1]
            if a < b:
                pair = a, b
            else:
                pair = b, a
            if pair in edges:
                k = edges[pair]
                adj[i].add(k)
                adj[k].add(i)
            else:
                edges[pair] = i
    return adj

#------------------------------------------------------------------------------
def facet_coloring(list facets):
    """Map-coloring greedy algorithm for the facets.

    Args:
        facets (list): list of numpy arrays of vertex indices forming the 
            facets; it may be necessary to eliminate the facet(s) corresponding
            to the boundaries
    
    Returns:
        array of int: for each facet, the color (0 to c-1) such that no 
            adjacent facets have the same color
    """
    cdef:
        list adj
        int V, c, i, u
        int [:] result
        char [:] available
    
    adj = facets_adjacency_matrix(facets)
    V = len(adj)
    result = np.full(V, -1, dtype=np.int32)
    result[0] = 0
    available = np.full(V, 0, dtype=np.uint8)

    for u in range(1, V):
        # Process all adjacent vertices and flag their colors as unavailable
        for i in adj[u]:
            if (result[i] != -1):
                available[result[i]] = 1
        # Find the first available color
        c = 0
        while c < V:
            if (available[c] == 0):
                break
            c += 1
        # Assign the found color
        result[u] = c 
        # Reset the values back to false for the next iteration
        for i in adj[u]:
            if (result[i] != -1):
                available[result[i]] = 0
                
    return np.array(result)

#------------------------------------------------------------------------------
def facet_normals(vertices, list facets):
    """Copmute the normal vector of the facets

    Args:
        vertices (nv-by-3 array): vertex positions
        facets (list): list of n-by-3 numpy arrays of vertex indices forming 
            the facets; the facets are grouped by the number of sides
    
    Returns:
        list of n-by-3 arrays: normal vectors for each facet grouped by the 
        number of sides
    """
    cdef:
        int n, k
        list normals
    normals = [np.empty((0, 3)), np.empty((0, 3)), np.empty((0, 3))]
    for n, f in enumerate(facets):
        if n < 3:
            continue
        x = vertices[f]
        L = np.zeros((f.shape[0], 3))
        for k in range(n):
            dx = x[:, (k+1) % n] - x[:, k]
            L += np.cross(x[:, k], dx)
        normals.append(L)
    return normals