# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['esdbclient', 'esdbclient.protos.Grpc']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.51.0,<2.0.0', 'protobuf>=4.21.0,<5.0.0', 'typing_extensions']

setup_kwargs = {
    'name': 'esdbclient',
    'version': '0.7',
    'description': 'Python gRPC Client for EventStoreDB',
    'long_description': '# Python gRPC Client for EventStoreDB\n\nThis package provides a Python gRPC client for\n[EventStoreDB](https://www.eventstore.com/). It has been\ndeveloped and tested to work with EventStoreDB LTS version 21.10,\nand with Python versions 3.7, 3.8, 3.9, 3.10, and 3.11.\n\nMethods have typing annotations, the static typing is checked\nwith mypy, and the test coverage is 100%.\n\nNot all the features of the EventStoreDB API are presented\nby this client in its current form, however many of the most\nuseful aspects are presented in an easy-to-use interface (see below).\nFor an example of usage, see the [eventsourcing-eventstoredb](\nhttps://github.com/pyeventsourcing/eventsourcing-eventstoredb) package.\n\n## Table of contents\n\n<!-- TOC -->\n* [Installation](#installation)\n* [Getting started](#getting-started)\n  * [Start EventStoreDB](#start-eventstoredb)\n  * [Stop EventStoreDB](#stop-eventstoredb)\n  * [Construct client](#construct-client)\n* [Streams](#streams)\n  * [Append events](#append-events)\n  * [Get current stream position](#get-current-stream-position)\n  * [Read stream events](#read-stream-events)\n  * [Read all recorded events](#read-all-recorded-events)\n  * [Get current commit position](#get-current-commit-position)\n* [Subscriptions](#subscriptions)\n  * [Catch-up subscriptions](#catch-up-subscriptions)\n  * [Persistent subscriptions](#persistent-subscriptions)\n* [Notes](#notes)\n  * [Regular expression filters](#regular-expression-filters)\n  * [The NewEvent class](#the-newevent-class)\n  * [The RecordedEvent class](#the-recordedevent-class)\n* [Contributors](#contributors)\n  * [Install Poetry](#install-poetry)\n  * [Setup for PyCharm users](#setup-for-pycharm-users)\n  * [Setup from command line](#setup-from-command-line)\n  * [Project Makefile commands](#project-makefile-commands)_\n<!-- TOC -->\n\n## Installation\n\nUse pip to install this package from\n[the Python Package Index](https://pypi.org/project/esdbclient/).\n\n    $ pip install esdbclient\n\nIt is recommended to install Python packages into a Python virtual environment.\n\n\n## Getting started\n\n### Start EventStoreDB\n\nUse Docker to run EventStoreDB from the official container image on DockerHub.\n\n    $ docker run -d --name my-eventstoredb -it -p 2113:2113 -p 1113:1113 eventstore/eventstore:21.10.2-buster-slim --insecure\n\nPlease note, this will start the server without SSL/TLS enabled, allowing\nonly "insecure" connections. This version of this Python client does not\nsupport SSL/TLS connections. A future version of this library will support\n"secure" connections.\n\n### Stop EventStoreDB\n\nUse Docker to stop and remove the EventStoreDB container.\n\n    $ docker stop my-eventstoredb\n\t$ docker rm my-eventstoredb\n\n\n### Construct client\n\nThe class `EsdbClient` can be constructed with a `uri` that indicates the\nhostname and port number of the EventStoreDB server.\n\n```python\nfrom esdbclient import EsdbClient\n\nclient = EsdbClient(uri=\'localhost:2113\')\n```\n\n## Streams\n\n### Append events\n\nThe client has an `append_events()` method, which can be used to append\nnew events to a "stream". A stream is a sequence of recorded events that\nis uniquely identified by a "stream name"\n\nThree arguments are required, `stream_name`, `expected_position`\nand `events`.\n\nThe `stream_name` argument is required, and is expected to be a Python\n`str` object that uniquely identifies the stream in the database.\n\nThe `expected_position` argument is required, is expected to be: either\na positive integer equal to the position in the stream of the last recorded\nevent in the stream (known as the "stream position"); or `None` if new events\nare being appended to a new stream.\n\nThe stream position sequences are zero-based, and so for example when a stream\nhas one recorded event, the stream position is `0`, and the correct value of the\n`expected_position` argument when appending the second new event should be `0`.\nThe correct value of the `expected_position` argument when appending the first\nevent of a new stream (a stream with zero recorded events) is `None`. That is,\nstreams are created by appending events with `expected_position=None`, and there\nis no way to create a stream without appending events.\n\nIf there is a mismatch between the given value of this argument and the\nactual stream position when the new events are recorded by the database,\nthen an `ExpectedPositionError` exception will be raised. This accomplishes\noptimistic concurrency control when appending new events.\n\nIf you wish to disable optimistic concurrency, set the\n`expected_position` to a negative integer.\n\nIf you need to get the current stream position, then use the `get_stream_position()`\nmethod (see below).\n\nThe `events` argument is required, and is expected to be a sequence of new\nevent objects to be appended to the named stream. The `NewEvent` class should\nbe used to construct new event objects (see below).\n\nPlease note, the append events operation is atomic, so that either all\nor none of the given new events will be recorded. By design, it is only\npossible with EventStoreDB to atomically record new events in one stream.\n\nIn the example below, a new event is appended to a new stream.\n\n```python\nfrom uuid import uuid4\n\nfrom esdbclient import NewEvent\n\n# Construct new event object.\nevent1 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'data1\',\n    metadata=b\'{}\'\n)\n\n# Define stream name.\nstream_name1 = str(uuid4())\n\n# Append list of events to new stream.\ncommit_position1 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=None,\n    events=[event1],\n)\n```\n\nIn the example below, two subsequent events are appended to an existing\nstream.\n\n```python\nevent2 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'data2\',\n    metadata=b\'{}\',\n)\nevent3 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'data3\',\n    metadata=b\'{}\',\n)\n\ncommit_position2 = client.append_events(\n    stream_name=stream_name1,\n    expected_position=0,\n    events=[event2, event3],\n)\n```\n\nIf the append operation is successful, this method returns an integer\nrepresenting the overall "commit position" as it was when the operation\nwas completed. Otherwise, an exception will be raised.\n\nA "commit position" is a monotonically increasing integer representing\nthe position of the recorded event in a "total order" of all recorded\nevents in the database across all streams. The sequence of commit positions\nis not gapless. It represents the position of the event record on disk, and\nthere are usually large differences between successive commits.\n\nThe "commit position" returned in this way can be used to wait for a\ndownstream component to have processed the newly appended events.\nFor example, after a user interface command that results in the recording\nof new events, and before a query is issued that depends on an eventually\nconsistent materialized view in a downstream component that would be stale\nif those newly appended events have not yet been processed, the user interface\ncan poll the downstream component, to see if it has processed an event at that\ncommit position, before executing a query for that materialized view.\n\n### Get current stream position\n\nThe client has a `get_stream_position()` method, which can be used to\nget the current "stream position" of a stream (the position in the\nstream of the last recorded event in that stream).\n\nThis method has a `stream_name` argument, which is required.\n\nThis method also takes an optional `timeout` argument, that\nis expected to be a Python `float`, which sets a deadline\nfor the completion of the gRPC operation.\n\nThe sequence of positions in a stream is gapless. It is zero-based,\nso that a stream with one recorded event has a current stream\nposition of `0`. The current stream position is `1` when a stream has\ntwo events, and it is `2` when there are events, and so on.\n\nIn the example below, the current stream position is obtained of the\nstream to which events were appended in the examples above.\nBecause the sequence of stream positions is zero-based, and because\nthree events were appended, so the current stream position is `2`.\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=stream_name1\n)\n\nassert stream_position == 2\n```\n\nIf a stream does not exist, the returned stream position value is `None`,\nwhich matches the required expected position when appending the first event\nof a new stream (see above).\n\n```python\nstream_position = client.get_stream_position(\n    stream_name=str(uuid4())\n)\n\nassert stream_position == None\n```\n\nThis method takes an optional argument `timeout` which is a float that sets\na deadline for the completion of the gRPC operation.\n\n\n### Read stream events\n\nThe client has a `read_stream_events()` method, which can be used to read\nthe events of a stream.\n\nThis method returns nn iterable object that yields recorded event objects.\nThese recorded event objects are instances of the `RecordedEvent` class (see below)\n\nThis method has one required argument, `stream_name`, which is the name of\nthe stream to be read. By default, the recorded events in the stream\nare returned in the order they were recorded.\n\nThe example below shows how to read the recorded events of a stream\nforwards from the start of the stream to the end of the stream. The\nname of a stream is given when calling the method. In this example,\nthe iterable response object is converted into a Python `list`, which\ncontains all the recorded event objects that were read from the stream.\n\n```python\nresponse = client.read_stream_events(\n    stream_name=stream_name1\n)\n\nevents = list(response)\n```\n\nNow that we have a list of event objects, we can check we got the\nthree events that were appended to the stream, and that they are\nordered exactly as they were appended.\n\n```python\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe method `read_stream_events()` also supports four optional arguments,\n`position`, `backwards`, `limit`, and `timeout`.\n\nThe optional `position` argument is an optional integer that can be used to indicate\nthe position in the stream from which to start reading. This argument is `None`\nby default, which means the stream will be read either from the start of the\nstream (the default behaviour), or from the end of the stream if `backwards` is\n`True` (see below). When reading a stream from a specific position in the stream, the\nrecorded event at that position WILL be included, both when reading forwards\nfrom that position, and when reading backwards from that position.\n\nThe optional argument `backwards` is a boolean, by default `False`, which means the\nstream will be read forwards by default, so that events are returned in the\norder they were appended, If `backwards` is `True`, the stream will be read\nbackwards, so that events are returned in reverse order.\n\nThe optional argument `limit` is an integer which limits the number of events that will\nbe returned. The default value is `sys.maxint`.\n\nThe optional argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation.\n\nThe example below shows how to read recorded events in a stream forwards from\na specific stream position to the end of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        position=1,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 1\nassert events[0].type == event2.type\nassert events[0].data == event2.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 2\nassert events[1].type == event3.type\nassert events[1].data == event3.data\n```\n\nThe example below shows how to read the recorded events in a stream backwards from\nthe end of the stream to the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        backwards=True,\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number (two) of the recorded events\nin a stream forwards from the start of the stream.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        limit=2,\n    )\n)\n\nassert len(events) == 2\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded\nevents in a stream backwards from a given stream position.\n\n```python\nevents = list(\n    client.read_stream_events(\n        stream_name=stream_name1,\n        position=2,\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\n### Read all recorded events\n\nThe method `read_all_events()` can be used to read all recorded events\nin the database in the order they were recorded. An iterable object of\nrecorded events is returned. This iterable object will stop when it has\nyielded the last recorded event.\n\nThe method `read_stream_events()` supports six optional arguments,\n`commit_position`, `backwards`, `filter_exclude`, `filter_include`, `limit`,\nand `timeout`.\n\nThe optional argument `position` is an optional integer that can be used to specify\nthe commit position from which to start reading. This argument is `None` by\ndefault, meaning that all the events will be read either from the start, or\nfrom the end if `backwards` is `True` (see below). Please note, if specified,\nthe specified position must be an actually existing commit position, because\nany other number will result in a server error (at least in EventStoreDB v21.10).\n\nPlease also note, when reading forwards from a specific commit position, the event\nat the specified position WILL be included. However, when reading backwards, the\nevent at the specified position will NOT be included. (This non-inclusive behaviour\nof excluding the specified commit position when reading all streams differs from the\nbehaviour when reading a named stream backwards from a specific stream position, I\'m\nnot sure why.)\n\nThe optional argument `backwards` is a boolean which is by default `False` meaning the\nevents will be read forwards by default, so that events are returned in the\norder they were committed, If `backwards` is `True`, the events will be read\nbackwards, so that events are returned in reverse order.\n\nThe optional argument `filter_exclude` is a sequence of regular expressions that\nmatch the type strings of recorded events that should not be included. By default,\nthis argument will match "system events", so that they will not be included.\nThis argument is ignored if `filter_include` is set to a non-empty sequence.\n\nThe optional argument `filter_include` is a sequence of regular expressions\nthat match the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_exclude` argument is ignored.\n\nPlease note, the filtering happens on the EventStoreDB server, and the\n`limit` argument is applied on the server after filtering. See below for\nmore information about filter regular expressions.\n\nThe optional argument `limit` is an integer which limits the number of events that will\nbe returned. The default value is `sys.maxint`.\n\nThe optional argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation.\n\nThe example below shows how to read all events in the database in the\norder they were recorded.\n\n```python\nevents = list(client.read_all_events())\n\nassert len(events) >= 3\n```\n\nThe example below shows how to read all recorded events from a particular commit position.\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position1\n    )\n)\n\nassert len(events) == 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 2\nassert events[2].type == event3.type\nassert events[2].data == event3.data\n```\n\nThe example below shows how to read all recorded events in reverse order.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True\n    )\n)\n\nassert len(events) >= 3\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n\nassert events[1].stream_name == stream_name1\nassert events[1].stream_position == 1\nassert events[1].type == event2.type\nassert events[1].data == event2.data\n\nassert events[2].stream_name == stream_name1\nassert events[2].stream_position == 0\nassert events[2].type == event1.type\nassert events[2].data == event1.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database forwards from a specific commit position. Please note, when reading\nall events forwards from a specific commit position, the event at the specified\nposition WILL be included.\n\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position1,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 0\nassert events[0].type == event1.type\nassert events[0].data == event1.data\n\nassert events[0].commit_position == commit_position1\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database backwards from the end. This gives the last recorded event.\n\n```python\nevents = list(\n    client.read_all_events(\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].stream_name == stream_name1\nassert events[0].stream_position == 2\nassert events[0].type == event3.type\nassert events[0].data == event3.data\n```\n\nThe example below shows how to read a limited number (one) of the recorded events\nin the database backwards from a specific commit position. Please note, when reading\nall events backwards from a specific commit position, the event at the specified\nposition WILL NOT be included.\n\n```python\nevents = list(\n    client.read_all_events(\n        commit_position=commit_position2,\n        backwards=True,\n        limit=1,\n    )\n)\n\nassert len(events) == 1\n\nassert events[0].commit_position < commit_position2\n```\n\n### Get current commit position\n\nThe method `get_commit_position()` can be used to get the current\ncommit position of the database.\n\n```python\ncommit_position = client.get_commit_position()\n```\n\nThis method takes an optional argument `timeout` which is a float that sets\na deadline for the completion of the gRPC operation.\n\nThis method can be useful to measure progress of a downstream component\nthat is processing all recorded events, by comparing the current commit\nposition with the recorded commit position of the last successfully processed\nevent in a downstream component.\n\nThe value of the `commit_position` argument when reading events either by using\nthe `read_all_events()` method or by using a catch-up subscription would usually\nbe determined by the recorded commit position of the last successfully processed\nevent in a downstream component.\n\n## Subscriptions\n\n### Catch-up subscriptions\n\nThe client has a `subscribe_all_events()` method, which can be used\nto start a "catch-up" subscription.\n\nMany catch-up subscriptions can be created, concurrently or\nsuccessively, and all will receive all the events they are\nsubscribed to receive.\n\nThis method returns an iterator object which yields recorded events,\nincluding events that are recorded after the subscription was created.\nThis iterator object will therefore not stop, unless the connection\nto the database is lost. The connection will be closed when the\niterator object is deleted from memory, which will happen when the\niterator object goes out of scope is explicitly deleted (see below),\nand the connection may be closed by the server.\n\nThis method takes an optional `commit_position` argument, which can be\nused to specify a commit position from which to subscribe for\nrecorded events. The default value is `None`, which means\nthe subscription will operate from the first recorded event\nin the database. If a commit position is given, it must match\nan actually existing commit position in the database. The events\nthat are obtained will not include the event recorded at that commit\nposition.\n\nThis method also takes three other optional arguments, `filter_exclude`,\n`filter_include`, and `timeout`.\n\nThe argument `filter_exclude` is a sequence of regular expressions matching\nthe type strings of recorded events that should be excluded. By default,\nthis argument will match "system events", so that they will not be included.\nThis argument is ignored if `filter_include` is set to a non-empty sequence.\n\nThe argument `filter_include` is a sequence of regular expressions\nmatching the type strings of recorded events that should be included. By\ndefault, this argument is an empty tuple. If this argument is set to a\nnon-empty sequence, the `filter_exclude` argument is ignored.\n\nPlease note, the filtering happens on the EventStoreDB server, and the\n`limit` argument is applied on the server after filtering. See below for\nmore information about filter regular expressions.\n\nThe argument `timeout` is a float which sets a deadline for the completion of\nthe gRPC operation. This probably isn\'t very useful, but is included for\ncompleteness and consistency with the other methods.\n\nThe example below shows how to subscribe to receive all recorded\nevents from a specific commit position. Three already-recorded\nevents are received, and then three new events are recorded, which\nare then received via the subscription.\n\n```python\n\n# Get the commit position (usually from database of materialised views).\ncommit_position = client.get_commit_position()\n\n# Append three events to another stream.\nstream_name2 = str(uuid4())\nevent4 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'data4\',\n    metadata=b\'{}\',\n)\nevent5 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'data5\',\n    metadata=b\'{}\',\n)\nevent6 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'data6\',\n    metadata=b\'{}\',\n)\nclient.append_events(\n    stream_name=stream_name2,\n    expected_position=None,\n    events=[event4, event5, event6],\n)\n\n# Subscribe from the commit position.\nsubscription = client.subscribe_all_events(\n    commit_position=commit_position\n)\n\n# Catch up by receiving the three events from the subscription.\nevents = []\nfor event in subscription:\n    events.append(event)\n    if event.data == event6.data and event.stream_name:\n        break\n\nassert events[0].data == event4.data\nassert events[1].data == event5.data\nassert events[2].data == event6.data\n\n\n# Append three more events.\nstream_name3 = str(uuid4())\nevent7 = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'data7\',\n    metadata=b\'{}\',\n)\nevent8 = NewEvent(\n    type=\'OrderUpdated\',\n    data=b\'data8\',\n    metadata=b\'{}\',\n)\nevent9 = NewEvent(\n    type=\'OrderDeleted\',\n    data=b\'data9\',\n    metadata=b\'{}\',\n)\n\nclient.append_events(\n    stream_name=stream_name3,\n    expected_position=None,\n    events=[event7, event8, event9],\n)\n\n# Receive the three new events from the same subscription.\nfor event in subscription:\n    # Check the stream name.\n    events.append(event)\n    if event.stream_name == stream_name3:\n        if event.data == event9.data:\n            break\n\nassert events[3].data == event7.data\nassert events[4].data == event8.data\nassert events[5].data == event9.data\n```\n\nCatch-up subscriptions are not registered in EventStoreDB (they are not\n"persistent" subscriptions). It is simply a streaming gRPC call which is\nkept open by the server, with newly recorded events sent to the client\nas the client iterates over the subscription. This kind of subscription\nis closed as soon as the subscription object goes out of memory.\n\n```python\n# End the subscription.\ndel subscription\n```\n\nPlease note, when processing events in a downstream component, the commit position of\nthe last successfully processed event is usefully recorded by the downstream component\nso that the commit position can be determined by the downstream component from its own\nrecorded when it is restarted. This commit position can be used to specify the commit\nposition from which to subscribe. Since this commit position represents the position of\nthe last successfully processed event in a downstream component, so it will be usual to\nwant the next event after this position, because that is the next event that needs to\nbe processed. When subscribing for events using a catchup-subscription\nin EventStoreDB, the event at the specified commit position will NOT be included in\nthe sequence of recorded events.\n\nTo accomplish "exactly once" processing of the events, the commit position\nof a recorded event should be recorded atomically and uniquely along with\nthe result of processing recorded events, for example in the same database\nas materialised views when implementing eventually-consistent CQRS, or in\nthe same database as a downstream analytics or reporting or archiving\napplication. This avoids "dual writing" in the processing of events.\n\nRecorded events received from a catch-up subscription cannot be acknowledged back\nto the EventStoreDB server (there is no need to do this). Acknowledging events is\nan aspect of "persistent subscriptions" (see below).\n\nThe subscription object might be used directly when processing events. It might\nalso be used within a thread dedicated to receiving events, with recorded events\nput on a queue for processing in a different thread. This package doesn\'t provide\nsuch thread or queue objects, you would need to do that yourself. Just make sure\nto reconstruct the subscription (and the queue) using your last recorded commit\nposition when resuming the subscription after an error, to be sure all events\nare processed once.\n\n### Persistent subscriptions\n\nThe method `create_subscription()` can be used to create a\n"persistent subscription" to EventStoreDB.\n\nThis method takes a required `group_name` argument, which is the\nname of a "group" of consumers of the subscription.\n\nThis method takes an optional `from_end` argument, which can be\nused to specify that the group of consumers of the subscription should\nonly receive events that were recorded after the subscription was created.\n\nThis method takes an optional `commit_position` argument, which can be\nused to specify a commit position from which the group of consumers of\nthe subscription should receive events. Please note, the recorded event\nat the specified commit position MAY be included in the recorded events\nreceived by the group of consumers.\n\nIf neither `from_end` or `position` are specified, the group of consumers\nof the subscription will receive all recorded events.\n\nThe method `create_subscription()` does not return a value, because\nrecorded events are obtained by the group of consumers of the subscription\nusing the `read_subscription()` method.\n\nIn the example below, a persistent subscription is created.\n\n```python\n# Create a persistent subscription.\ngroup_name = f"group-{uuid4()}"\nclient.create_subscription(group_name=group_name)\n```\n\nThe method `read_subscription()` can be used by a group of consumers to receive\nrecorded events from a persistent subscription created using `create_subscription`.\n\nThis method takes a required `group_name` argument, which is\nthe name of a "group" of consumers of the subscription specified\nwhen `create_subscription()` was called.\n\nThis method returns a 2-tuple: a "read request" object and a "read response" object.\n\n```python\nread_req, read_resp = client.read_subscription(group_name=group_name)\n```\n\nThe "read response" object is an iterator that yields recorded events from\nthe specified commit position.\n\nThe "read request" object has an `ack()` method that can be used by a consumer\nin a group to acknowledge to the server that it has received and successfully\nprocessed a recorded event. This will prevent that recorded event being received\nby another consumer in the same group. The `ack()` method takes an `event_id`\nargument, which is the ID of the recorded event that has been received.\n\nThe example below iterates over the "read response" object, and calls `ack()`\non the "read response" object. The for loop breaks when we have received\nthe last event, so that we can continue with the examples below.\n\n```python\nevents = []\nfor event in read_resp:\n    events.append(event)\n\n    # Acknowledge the received event.\n    read_req.ack(event_id=event.id)\n\n    # Break when the last event has been received.\n    if event.stream_name == stream_name3:\n        if event.data == event9.data:\n            break\n```\n\nThe received events are the events we appended above.\n\n```python\nassert events[-9].data == event1.data\nassert events[-8].data == event2.data\nassert events[-7].data == event3.data\nassert events[-6].data == event4.data\nassert events[-5].data == event5.data\nassert events[-4].data == event6.data\nassert events[-3].data == event7.data\nassert events[-2].data == event8.data\nassert events[-1].data == event9.data\n```\n\nThe "read request" object also has an `nack()` method that can be used by a consumer\nin a group to acknowledge to the server that it has failed successfully to\nprocess a recorded event. This will allow that recorded event to be received\nby this or another consumer in the same group.\n\nIt might be more useful to encapsulate the request and response objects and to iterate\nover the "read response" in a separate thread, to call back to a handler function when\na recorded event is received, and call `ack()` if the handler does not raise an\nexception, and to call `nack()` if an exception is raised. The example below shows how\nthis might be done.\n\n```python\nfrom threading import Thread\n\n\nclass SubscriptionReader:\n    def __init__(self, client, group_name, callback):\n        self.client = client\n        self.group_name = group_name\n        self.callback = callback\n        self.thread = Thread(target=self.read_subscription, daemon=True)\n        self.error = None\n\n    def start(self):\n        self.thread.start()\n\n    def join(self):\n        self.thread.join()\n\n    def read_subscription(self):\n        req, resp = self.client.read_subscription(group_name=self.group_name)\n        for event in resp:\n            try:\n                self.callback(event)\n            except Exception as e:\n                # req.nack(event.id)  # not yet implemented....\n                self.error = e\n                break\n            else:\n                req.ack(event.id)\n\n\n# Create another persistent subscription.\ngroup_name = f"group-{uuid4()}"\nclient.create_subscription(group_name=group_name)\n\nevents = []\n\ndef handle_event(event):\n    events.append(event)\n    print("Event:", event.stream_name, event.data)\n    if event.stream_name == stream_name3:\n        if event.data == event9.data:\n            raise Exception()\n\n\nreader = SubscriptionReader(\n    client=client,\n    group_name=group_name,\n    callback=handle_event\n)\n\nreader.start()\nreader.join()\n\nassert events[-1].data == event9.data\n```\n\nPlease note, when processing events in a downstream component, the commit position of\nthe last successfully processed event is usefully recorded by the downstream component\nso that the commit position can be determined by the downstream component from its own\nrecorded when it is restarted. This commit position can be used to specify the commit\nposition from which to subscribe. Since this commit position represents the position of\nthe last successfully processed event in a downstream component, so it will be usual to\nwant to read from the next event after this position, because that is the next event\nthat needs to be processed. However, when subscribing for events using a persistent\nsubscription in EventStoreDB, the event at the specified commit position MAY be returned\nas the first event in the received sequence of recorded events, and so it may\nbe necessary to check the commit position of the received events and to discard\nany  recorded event object that has a commit position equal to the commit position\nspecified in the request.\n\nWhilst there are some advantages of persistent subscriptions, by tracking in the\nupstream server the position in the commit sequence of events that have been processed,\nthere is a danger of "dual writing" in the consumption of events. The danger is that if\nan event is successfully processed but then the acknowledgment fails, the event may be\nreceived more than once. On the other hand, if the acknowledgment is successful but\nthen the processing fails, the event may effectively not be been processed. By either\nprocessing an events more than once, or failing to process an event, the resulting state\nof the processing of the recorded events might be inaccurate, or possibly\ninconsistent, and perhaps catastrophically so. Any relatively minor consequences may or\nmay not matter in your situation. But sometimes inconsistencies may halt processing\nuntil the issue is resolved. You can avoid "dual writing" in the consumption of events\nby atomically recording the commit position of an event that has been processed along\nwith the results of processing that event (that is, with both things being recorded in\nthe same transaction), and making these records unique so that transactions will be\nrolled back preventing the results of reprocessing the event being committed.\n\n## Notes\n\n### Regular expression filters\n\nThe `filter_exclude` and `filter_include` arguments in `read_all_events()` and\n`subscribe_all_events()` are applied to the `type` attribute of recorded events.\n\nThe default value of the `filter_exclude` arguments is designed to exclude\nEventStoreDB "system events", which otherwise would be included. System\nevents, by convention in EventStoreDB, all have `type` strings that\nstart with the `$` sign.\n\nPlease note, characters that have a special meaning in regular expressions\nwill need to be escaped (with double-backslash) when matching these characters\nin type strings.\n\nFor example, to match EventStoreDB system events, use the sequence `[\'\\\\$.*\']`.\nPlease note, the constant `ESDB_EVENTS_REGEX` is set to `\'\\\\$.*\'`. You\ncan import this value (`from esdbclient import ESDB_EVENTS_REGEX`) and use\nit when building longer sequences of regular expressions. For example,\nto exclude system events and snapshots, you might use the sequence\n`[ESDB_EVENTS_REGEX, \'.*Snapshot\']` as the value of the `filter_exclude`\nargument.\n\n\n### The NewEvent class\n\nThe `NewEvent` class can be used to define new events.\n\nThe attribute `type` is a unicode string, used to specify the type of the event\nto be recorded.\n\nThe attribute `data` is a byte string, used to specify the data of the event\nto be recorded. Please note, in this version of this Python client,\nwriting JSON event data to EventStoreDB isn\'t supported, but it might be in\na future version.\n\nThe attribute `metadata` is a byte string, used to specify metadata for the event\nto be recorded.\n\n```python\nnew_event = NewEvent(\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n)\n```\n\n### The RecordedEvent class\n\nThe `RecordedEvent` class is used when reading recorded events.\n\nThe attribute `type` is a unicode string, used to indicate the type of the event\nthat was recorded.\n\nThe attribute `data` is a byte string, used to indicate the data of the event\nthat was recorded.\n\nThe attribute `metadata` is a byte string, used to indicate metadata for the event\nthat was recorded.\n\nThe attribute `stream_name` is a unicode string, used to indicate the type of\nthe name of the stream in which the event was recorded.\n\nThe attribute `stream_position` is an integer, used to indicate\nthe position in the stream at which the event was recorded.\n\nThe attribute `commit_position` is an integer, used to indicate\nthe position in total order of all recorded events at which the\nevent was recorded.\n\n```python\nfrom esdbclient.events import RecordedEvent\n\nrecorded_event = RecordedEvent(\n    id=uuid4(),\n    type=\'OrderCreated\',\n    data=b\'{}\',\n    metadata=b\'{}\',\n    stream_name=\'stream1\',\n    stream_position=0,\n    commit_position=512,\n)\n```\n\n## Contributors\n\n### Install Poetry\n\nThe first thing is to check you have Poetry installed.\n\n    $ poetry --version\n\nIf you don\'t, then please [install Poetry](https://python-poetry.org/docs/#installing-with-the-official-installer).\n\n    $ curl -sSL https://install.python-poetry.org | python3 -\n\nIt will help to make sure Poetry\'s bin directory is in your `PATH` environment variable.\n\nBut in any case, make sure you know the path to the `poetry` executable. The Poetry\ninstaller tells you where it has been installed, and how to configure your shell.\n\nPlease refer to the [Poetry docs](https://python-poetry.org/docs/) for guidance on\nusing Poetry.\n\n### Setup for PyCharm users\n\nYou can easily obtain the project files using PyCharm (menu "Git > Clone...").\nPyCharm will then usually prompt you to open the project.\n\nOpen the project in a new window. PyCharm will then usually prompt you to create\na new virtual environment.\n\nCreate a new Poetry virtual environment for the project. If PyCharm doesn\'t already\nknow where your `poetry` executable is, then set the path to your `poetry` executable\nin the "New Poetry Environment" form input field labelled "Poetry executable". In the\n"New Poetry Environment" form, you will also have the opportunity to select which\nPython executable will be used by the virtual environment.\n\nPyCharm will then create a new Poetry virtual environment for your project, using\na particular version of Python, and also install into this virtual environment the\nproject\'s package dependencies according to the `pyproject.toml` file, or the\n`poetry.lock` file if that exists in the project files.\n\nYou can add different Poetry environments for different Python versions, and switch\nbetween them using the "Python Interpreter" settings of PyCharm. If you want to use\na version of Python that isn\'t installed, either use your favourite package manager,\nor install Python by downloading an installer for recent versions of Python directly\nfrom the [Python website](https://www.python.org/downloads/).\n\nOnce project dependencies have been installed, you should be able to run tests\nfrom within PyCharm (right-click on the `tests` folder and select the \'Run\' option).\n\nBecause of a conflict between pytest and PyCharm\'s debugger and the coverage tool,\nyou may need to add ``--no-cov`` as an option to the test runner template. Alternatively,\njust use the Python Standard Library\'s ``unittest`` module.\n\nYou should also be able to open a terminal window in PyCharm, and run the project\'s\nMakefile commands from the command line (see below).\n\n### Setup from command line\n\nObtain the project files, using Git or suitable alternative.\n\nIn a terminal application, change your current working directory\nto the root folder of the project files. There should be a Makefile\nin this folder.\n\nUse the Makefile to create a new Poetry virtual environment for the\nproject and install the project\'s package dependencies into it,\nusing the following command.\n\n    $ make install-packages\n\nIt\'s also possible to also install the project in \'editable mode\'.\n\n    $ make install\n\nPlease note, if you create the virtual environment in this way, and then try to\nopen the project in PyCharm and configure the project to use this virtual\nenvironment as an "Existing Poetry Environment", PyCharm sometimes has some\nissues (don\'t know why) which might be problematic. If you encounter such\nissues, you can resolve these issues by deleting the virtual environment\nand creating the Poetry virtual environment using PyCharm (see above).\n\n### Project Makefile commands\n\nYou can start EventStoreDB using the following command.\n\n    $ make start-eventstoredb\n\nYou can run tests using the following command (needs EventStoreDB to be running).\n\n    $ make test\n\nYou can stop EventStoreDB using the following command.\n\n    $ make stop-eventstoredb\n\nYou can check the formatting of the code using the following command.\n\n    $ make lint\n\nYou can reformat the code using the following command.\n\n    $ make fmt\n\nTests belong in `./tests`. Code-under-test belongs in `./esdbclient`.\n\nEdit package dependencies in `pyproject.toml`. Update installed packages (and the\n`poetry.lock` file) using the following command.\n\n    $ make update-packages\n',
    'author': 'John Bywater',
    'author_email': 'john.bywater@appropriatesoftware.net',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pyeventsourcing/esdbclient',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
