"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_s3_1 = require("@aws-cdk/aws-s3");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../../core/lib");
const ec2 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'can create Windows user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forWindows();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>command1\ncommand2</powershell>');
        test.done();
    },
    'can create Windows user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'onexit1\n' +
            'onexit2\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'command2\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can create Windows with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addSignalOnExitCommand(resource);
        userData.addCommands('command1');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'cfn-signal --stack Default --resource RESOURCE1989552F --region ${Token[AWS.Region.4]} --success ($success.ToString().ToLower())\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can windows userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forWindows();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.bat',
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.bat',
            localFile: 'c:\\test\\location\\otherScript.bat',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>mkdir (Split-Path -Path \'C:/temp/filename.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test\' -key \'filename.bat\' -file \'C:/temp/filename.bat\' -ErrorAction Stop\n' +
            'mkdir (Split-Path -Path \'c:\\test\\location\\otherScript.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test2\' -key \'filename2.bat\' -file \'c:\\test\\location\\otherScript.bat\' -ErrorAction Stop</powershell>');
        test.done();
    },
    'can windows userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename.bat',
        });
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename2.bat',
            arguments: 'arg1 arg2 -arg $variable',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>&\'C:\\test\\filename.bat\' undefined\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename.bat"\' -ErrorAction Stop }\n' +
            '&\'C:\\test\\filename2.bat\' arg1 arg2 -arg $variable\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename2.bat"\' -ErrorAction Stop }</powershell>');
        test.done();
    },
    'can create Linux user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\ncommand1\ncommand2');
        test.done();
    },
    'can create Linux user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            'onexit1\n' +
            'onexit2\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1\n' +
            'command2');
        test.done();
    },
    'can create Linux with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1');
        userData.addSignalOnExitCommand(resource);
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Default --resource RESOURCE1989552F --region ${Token[AWS.Region.4]} -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1');
        test.done();
    },
    'can linux userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forLinux();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh',
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.sh',
            localFile: 'c:\\test\\location\\otherScript.sh',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'mkdir -p $(dirname \'/tmp/filename.sh\')\n' +
            'aws s3 cp \'s3://test/filename.sh\' \'/tmp/filename.sh\'\n' +
            'mkdir -p $(dirname \'c:\\test\\location\\otherScript.sh\')\n' +
            'aws s3 cp \'s3://test2/filename2.sh\' \'c:\\test\\location\\otherScript.sh\'');
        test.done();
    },
    'can linux userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        });
        userData.addExecuteFileCommand({
            filePath: '/test/filename2.sh',
            arguments: 'arg1 arg2 -arg $variable',
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'set -e\n' +
            'chmod +x \'/tmp/filename.sh\'\n' +
            '\'/tmp/filename.sh\' undefined\n' +
            'set -e\n' +
            'chmod +x \'/test/filename2.sh\'\n' +
            '\'/test/filename2.sh\' arg1 arg2 -arg $variable');
        test.done();
    },
    'can create Custom user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('Some\nmultiline\ncontent');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, 'Some\nmultiline\ncontent');
        test.done();
    },
    'Custom user data throws when adding on exit commands'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addOnExitCommands('a command goes here'));
        test.done();
    },
    'Custom user data throws when adding signal command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addSignalOnExitCommand(resource));
        test.done();
    },
    'Custom user data throws when downloading file'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.custom('');
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        // WHEN
        // THEN
        test.throws(() => userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh',
        }));
        test.done();
    },
    'Custom user data throws when executing file'(test) {
        // GIVEN
        const userData = ec2.UserData.custom('');
        // WHEN
        // THEN
        test.throws(() => userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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