"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('can make and use a Windows image', () => {
    // WHEN
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    });
    // THEN
    const details = image.getImage(stack);
    expect(details.imageId).toEqual('ami-1234');
    expect(details.osType).toEqual(ec2.OperatingSystemType.WINDOWS);
});
test('can make and use a Generic SSM image', () => {
    // WHEN
    const image = new ec2.GenericSSMParameterImage('testParam', ec2.OperatingSystemType.LINUX);
    // THEN
    const details = image.getImage(stack);
    expect(details.imageId).toContain('TOKEN');
    expect(details.osType).toEqual(ec2.OperatingSystemType.LINUX);
});
test('WindowsImage retains userdata if given', () => {
    // WHEN
    const ud = ec2.UserData.forWindows();
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    }, {
        userData: ud,
    });
    // THEN
    const details = image.getImage(stack);
    expect(details.userData).toEqual(ud);
});
test('WindowsImage creates UserData if not given', () => {
    // WHEN
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    });
    // THEN
    const details = image.getImage(stack);
    expect(isWindowsUserData(details.userData)).toBeTruthy();
});
test('LookupMachineImage default search', () => {
    // GIVEN
    // WHEN
    new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'] }).getImage(stack);
    // THEN
    const missing = app.synth().manifest.missing || [];
    expect(missing).toEqual([
        {
            key: 'ami:account=1234:filters.image-type.0=machine:filters.name.0=bla*:filters.state.0=available:owners.0=amazon:region=testregion',
            props: {
                account: '1234',
                region: 'testregion',
                owners: ['amazon'],
                filters: {
                    'name': ['bla*'],
                    'state': ['available'],
                    'image-type': ['machine'],
                },
            },
            provider: 'ami',
        },
    ]);
});
test('LookupMachineImage creates correct type of UserData', () => {
    // WHEN
    const linuxDetails = new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'] }).getImage(stack);
    const windowsDetails = new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'], windows: true }).getImage(stack);
    // THEN
    expect(isWindowsUserData(windowsDetails.userData)).toBeTruthy();
    expect(isLinuxUserData(linuxDetails.userData)).toBeTruthy();
});
function isWindowsUserData(ud) {
    return ud.render().indexOf('powershell') > -1;
}
function isLinuxUserData(ud) {
    return ud.render().indexOf('bash') > -1;
}
//# sourceMappingURL=data:application/json;base64,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