"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'instance is created correctly'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.COMPUTE6_GRAVITON2, lib_1.InstanceSize.LARGE),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 'c6g.large',
        }));
        test.done();
    },
    'instance is created with source/dest check switched off'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
            sourceDestCheck: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
            SourceDestCheck: false,
        }));
        test.done();
    },
    'instance is grantable'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const param = new aws_ssm_1.StringParameter(stack, 'Param', { stringValue: 'Foobar' });
        const instance = new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        });
        // WHEN
        param.grantRead(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/',
                                    {
                                        Ref: 'Param165332EC',
                                    },
                                ],
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    blockDeviceMappings: {
        'can set blockDeviceMappings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        mappingEnabled: true,
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            volumeType: lib_1.EbsDeviceVolumeType.IO1,
                            iops: 5000,
                        }),
                    }, {
                        deviceName: 'ebs-snapshot',
                        mappingEnabled: false,
                        volume: lib_1.BlockDeviceVolume.ebsFromSnapshot('snapshot-id', {
                            volumeSize: 500,
                            deleteOnTermination: false,
                            volumeType: lib_1.EbsDeviceVolumeType.SC1,
                        }),
                    }, {
                        deviceName: 'ephemeral',
                        volume: lib_1.BlockDeviceVolume.ephemeral(0),
                    }],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                BlockDeviceMappings: [
                    {
                        DeviceName: 'ebs',
                        Ebs: {
                            DeleteOnTermination: true,
                            Encrypted: true,
                            Iops: 5000,
                            VolumeSize: 15,
                            VolumeType: 'io1',
                        },
                    },
                    {
                        DeviceName: 'ebs-snapshot',
                        Ebs: {
                            DeleteOnTermination: false,
                            SnapshotId: 'snapshot-id',
                            VolumeSize: 500,
                            VolumeType: 'sc1',
                        },
                        NoDevice: {},
                    },
                    {
                        DeviceName: 'ephemeral',
                        VirtualName: 'ephemeral0',
                    },
                ],
            }));
            test.done();
        },
        'throws if ephemeral volumeIndex < 0'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // THEN
            test.throws(() => {
                new lib_1.Instance(stack, 'Instance', {
                    vpc,
                    machineImage: new lib_1.AmazonLinuxImage(),
                    instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                    blockDevices: [{
                            deviceName: 'ephemeral',
                            volume: lib_1.BlockDeviceVolume.ephemeral(-1),
                        }],
                });
            }, /volumeIndex must be a number starting from 0/);
            test.done();
        },
        'throws if volumeType === IO1 without iops'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // THEN
            test.throws(() => {
                new lib_1.Instance(stack, 'Instance', {
                    vpc,
                    machineImage: new lib_1.AmazonLinuxImage(),
                    instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                    blockDevices: [{
                            deviceName: 'ebs',
                            volume: lib_1.BlockDeviceVolume.ebs(15, {
                                deleteOnTermination: true,
                                encrypted: true,
                                volumeType: lib_1.EbsDeviceVolumeType.IO1,
                            }),
                        }],
                });
            }, /ops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            test.done();
        },
        'warning if iops without volumeType'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            const instance = new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            iops: 5000,
                        }),
                    }],
            });
            // THEN
            test.deepEqual(instance.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
            test.deepEqual(instance.node.metadata[0].data, 'iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            test.done();
        },
        'warning if iops and volumeType !== IO1'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            const instance = new lib_1.Instance(stack, 'Instance', {
                vpc,
                machineImage: new lib_1.AmazonLinuxImage(),
                instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
                blockDevices: [{
                        deviceName: 'ebs',
                        volume: lib_1.BlockDeviceVolume.ebs(15, {
                            deleteOnTermination: true,
                            encrypted: true,
                            volumeType: lib_1.EbsDeviceVolumeType.GP2,
                            iops: 5000,
                        }),
                    }],
            });
            // THEN
            test.deepEqual(instance.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
            test.deepEqual(instance.node.metadata[0].data, 'iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            test.done();
        },
    },
    'instance can be created with Private IP Address'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.Instance(stack, 'Instance', {
            vpc,
            machineImage: new lib_1.AmazonLinuxImage(),
            instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
            privateIpAddress: '10.0.0.2',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            InstanceType: 't3.large',
            PrivateIpAddress: '10.0.0.2',
        }));
        test.done();
    },
});
test('add CloudFormation Init to instance', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const vpc = new lib_1.Vpc(stack, 'VPC');
    new lib_1.Instance(stack, 'Instance', {
        vpc,
        machineImage: new lib_1.AmazonLinuxImage(),
        instanceType: lib_1.InstanceType.of(lib_1.InstanceClass.T3, lib_1.InstanceSize.LARGE),
        init: lib_1.CloudFormationInit.fromElements(lib_1.InitCommand.shellCommand('echo hello')),
    });
    // THEN
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': {
                'Fn::Join': ['', [
                        assert_1.stringLike('#!/bin/bash\n# fingerprint: *\n(\n  set +e\n  /opt/aws/bin/cfn-init -v --region '),
                        { Ref: 'AWS::Region' },
                        ' --stack ',
                        { Ref: 'AWS::StackName' },
                        ' --resource InstanceC1063A87 -c default\n  /opt/aws/bin/cfn-signal -e $? --region ',
                        { Ref: 'AWS::Region' },
                        ' --stack ',
                        { Ref: 'AWS::StackName' },
                        ' --resource InstanceC1063A87\n  cat /var/log/cfn-init.log >&2\n)',
                    ]],
            },
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
                Effect: 'Allow',
                Resource: { Ref: 'AWS::StackId' },
            }),
            Version: '2012-10-17',
        },
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT5M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
//# sourceMappingURL=data:application/json;base64,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