"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'peering between two security groups does not recursive infinitely'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SG1', { vpc });
        const sg2 = new lib_1.SecurityGroup(stack, 'SG2', { vpc });
        const conn1 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const conn2 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg2] }));
        // WHEN
        conn1.connections.allowTo(conn2, lib_1.Port.tcp(80), 'Test');
        // THEN -- it finishes!
        test.done();
    },
    '(imported) SecurityGroup can be used as target of .allowTo()'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowTo(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            DestinationSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: 'Connect there',
            FromPort: 0,
            GroupId: 'sg-12345',
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            ToPort: 65535,
        }));
        test.done();
    },
    'security groups added to connections after rule still gets rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowFromAnyIpv4(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Default/SecurityGroup1',
            SecurityGroupIngress: [
                {
                    Description: 'from 0.0.0.0/0:88',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Default/SecurityGroup2',
            SecurityGroupIngress: [
                {
                    Description: 'from 0.0.0.0/0:88',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp',
                },
            ],
        }));
        test.done();
    },
    'when security groups are added to target they also get the rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const sg3 = new lib_1.SecurityGroup(stack, 'SecurityGroup3', { vpc, allowAllOutbound: false });
        const connections1 = new lib_1.Connections({ securityGroups: [sg1] });
        const connections2 = new lib_1.Connections({ securityGroups: [sg2] });
        const connectable = new SomethingConnectable(connections2);
        // WHEN
        connections1.allowTo(connectable, lib_1.Port.tcp(88));
        connections2.addSecurityGroup(sg3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup23BE86BB7', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup3E5E374B9', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        test.done();
    },
    'multiple security groups allows internally between them'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowInternally(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            GroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowFrom'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        app.synth();
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowTo'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowTo(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        app.synth();
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
            DestinationSecurityGroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
        }));
        test.done();
    },
    'can establish multiple cross-stack SGs'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1a = new lib_1.SecurityGroup(stack1, 'SecurityGroupA', { vpc: vpc1, allowAllOutbound: false });
        const sg1b = new lib_1.SecurityGroup(stack1, 'SecurityGroupB', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1a, lib_1.Port.tcp(100));
        sg2.connections.allowFrom(sg1b, lib_1.Port.tcp(100));
        // THEN -- both egress rules are in Stack2
        app.synth();
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupAED40ADC5GroupId1D10C76A' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupB04591F90GroupIdFA7208D5' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        test.done();
    },
    'Imported SecurityGroup does not create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            SourceSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroupEgress'));
        test.done();
    },
    'Imported SecurityGroup with allowAllOutbound: false DOES create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345', {
            allowAllOutbound: false,
        });
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            SourceSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            IpProtocol: 'tcp',
            Description: 'Connect there',
            FromPort: 0,
            GroupId: 'sg-12345',
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            ToPort: 65535,
        }));
        test.done();
    },
});
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
//# sourceMappingURL=data:application/json;base64,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