"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const fs = require("fs");
const os = require("os");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
const cfn_init_internal_1 = require("../lib/private/cfn-init-internal");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('InitCommand', () => {
    test('throws error on empty argv command', () => {
        expect(() => { ec2.InitCommand.argvCommand([]); }).toThrow();
    });
    test('auto-generates an indexed command key if none is provided', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh');
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered['000']).toBeDefined();
    });
    test('renders a minimalist template when no options are defined', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh');
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '000': { command: ['/bin/sh'] },
        });
    });
    test('creates a shell command with all provided options', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh', {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            command_0: {
                command: ['/bin/sh'],
                env: { SECRETS_FILE: '/tmp/secrets' },
                cwd: '/home/myUser',
                test: 'test -d /home/myUser',
                ignoreErrors: false,
                waitAfterCompletion: 7200,
            },
        });
    });
    test('creates an argv command with all provided options', () => {
        // GIVEN
        const command = ec2.InitCommand.argvCommand(['/bin/sh', '-c', 'doStuff'], {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // WHEN
        const rendered = getElementConfig(command, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            command_0: {
                command: ['/bin/sh', '-c', 'doStuff'],
                env: { SECRETS_FILE: '/tmp/secrets' },
                cwd: '/home/myUser',
                test: 'test -d /home/myUser',
                ignoreErrors: false,
                waitAfterCompletion: 7200,
            },
        });
    });
    test('errors when waitAfterCompletion is specified for Linux systems', () => {
        // GIVEN
        const command = ec2.InitCommand.shellCommand('/bin/sh', {
            key: 'command_0',
            env: { SECRETS_FILE: '/tmp/secrets' },
            cwd: '/home/myUser',
            testCmd: 'test -d /home/myUser',
            ignoreErrors: false,
            waitAfterCompletion: ec2.InitCommandWaitDuration.of(core_1.Duration.hours(2)),
        });
        // THEN
        expect(() => {
            command._bind(defaultOptions(cfn_init_internal_1.InitPlatform.LINUX));
        }).toThrow(/'waitAfterCompletion' is only valid for Windows/);
    });
});
describe('InitFile', () => {
    test('fromString creates inline content', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', 'My content');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'My content',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromString creates inline content from base64-encoded content', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', Buffer.from('Hello').toString('base64'), {
            base64Encoded: true,
        });
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'SGVsbG8=',
                encoding: 'base64',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('mode, user, group settings not allowed for Windows', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', 'My content', {
            group: 'root',
            owner: 'root',
            mode: '000644',
        });
        // WHEN
        expect(() => {
            file._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Owner, group, and mode options not supported for Windows.');
    });
    test('file renders properly on Windows', () => {
        // GIVEN
        const file = ec2.InitFile.fromString('/tmp/foo', 'My content');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'My content',
                encoding: 'plain',
            },
        });
    });
    test('symlink throws an error if mode is set incorrectly', () => {
        expect(() => {
            ec2.InitFile.symlink('/tmp/foo', '/tmp/bar', {
                mode: '000644',
            });
        }).toThrow('File mode for symlinks must begin with 120XXX');
    });
    test('symlink sets mode is not set', () => {
        // GIVEN
        const file = ec2.InitFile.symlink('/tmp/foo', '/tmp/bar');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: '/tmp/bar',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '120644',
            },
        });
    });
    test('fromFileInline renders the file contents inline', () => {
        // GIVEN
        const tmpFilePath = createTmpFileWithContent('Hello World!');
        const file = ec2.InitFile.fromFileInline('/tmp/foo', tmpFilePath);
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'Hello World!',
                encoding: 'plain',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromObject renders the contents inline as an object', () => {
        // GIVEN
        const content = {
            version: '1234',
            secretsFile: '/tmp/secrets',
        };
        const file = ec2.InitFile.fromObject('/tmp/foo', content);
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: {
                    version: '1234',
                    secretsFile: '/tmp/secrets',
                },
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromFileInline respects the base64 flag', () => {
        // GIVEN
        const tmpFilePath = createTmpFileWithContent('Hello');
        const file = ec2.InitFile.fromFileInline('/tmp/foo', tmpFilePath, {
            base64Encoded: true,
        });
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                content: 'SGVsbG8=',
                encoding: 'base64',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
    test('fromUrl uses the provided URL as a source', () => {
        // GIVEN
        const file = ec2.InitFile.fromUrl('/tmp/foo', 'https://aws.amazon.com/');
        // WHEN
        const rendered = getElementConfig(file, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': {
                source: 'https://aws.amazon.com/',
                owner: 'root',
                group: 'root',
                mode: '000644',
            },
        });
    });
});
describe('InitGroup', () => {
    test('renders without a group id', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon');
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ amazon: {} });
    });
    test('renders with a group id', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon', 42);
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ amazon: { gid: 42 } });
    });
    test('groups are not supported for Windows', () => {
        // GIVEN
        const group = ec2.InitGroup.fromName('amazon');
        // WHEN
        expect(() => {
            group._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Init groups are not supported on Windows');
    });
});
describe('InitUser', () => {
    test('fromName accepts just a name to create a user', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1');
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({ sysuser1: {} });
    });
    test('renders with all options present', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1', {
            userId: 42,
            homeDir: '/home/sysuser1',
            groups: ['amazon'],
        });
        // WHEN
        const rendered = getElementConfig(group, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            sysuser1: {
                uid: 42,
                homeDir: '/home/sysuser1',
                groups: ['amazon'],
            },
        });
    });
    test('users are not supported for Windows', () => {
        // GIVEN
        const group = ec2.InitUser.fromName('sysuser1');
        // WHEN
        expect(() => {
            group._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Init users are not supported on Windows');
    });
});
describe('InitPackage', () => {
    test('rpm auto-generates a name if none is provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            rpm: {
                '000': ['https://example.com/rpm/mypkg.rpm'],
            },
        });
    });
    test('rpm uses name if provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm', { key: 'myPkg' });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            rpm: {
                myPkg: ['https://example.com/rpm/mypkg.rpm'],
            },
        });
    });
    test('rpm is not supported for Windows', () => {
        // GIVEN
        const pkg = ec2.InitPackage.rpm('https://example.com/rpm/mypkg.rpm');
        // THEN
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Windows only supports the MSI package type');
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s accepts a package without versions', (pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            [pkgType]: { httpd: [] },
        });
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s accepts a package with versions', (pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd', { version: ['1.0', '2.0'] });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            [pkgType]: { httpd: ['1.0', '2.0'] },
        });
    });
    test.each([
        ['yum', ec2.InitPackage.yum],
        ['rubygems', ec2.InitPackage.rubyGem],
        ['python', ec2.InitPackage.python],
        ['apt', ec2.InitPackage.apt],
    ])('%s is not supported on Windows', (_pkgType, fn) => {
        // GIVEN
        const pkg = fn('httpd');
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.WINDOWS));
        }).toThrow('Windows only supports the MSI package type');
    });
    test('msi auto-generates a name if none is provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi');
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            msi: {
                '000': 'https://example.com/rpm/mypkg.msi',
            },
        });
    });
    test('msi uses name if provided', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi', { key: 'myPkg' });
        // WHEN
        const rendered = getElementConfig(pkg, cfn_init_internal_1.InitPlatform.WINDOWS);
        // THEN
        expect(rendered).toEqual({
            msi: {
                myPkg: 'https://example.com/rpm/mypkg.msi',
            },
        });
    });
    test('msi is not supported for Linux', () => {
        // GIVEN
        const pkg = ec2.InitPackage.msi('https://example.com/rpm/mypkg.msi');
        // THEN
        expect(() => {
            pkg._bind(defaultOptions(cfn_init_internal_1.InitPlatform.LINUX));
        }).toThrow('MSI installers are only supported on Windows systems.');
    });
});
describe('InitService', () => {
    test.each([
        ['Linux', 'sysvinit', cfn_init_internal_1.InitPlatform.LINUX],
        ['Windows', 'windows', cfn_init_internal_1.InitPlatform.WINDOWS],
    ])('enable always sets enabled and running to true for %s', (_platform, key, platform) => {
        // GIVEN
        const service = ec2.InitService.enable('httpd');
        // WHEN
        const rendered = service._bind(defaultOptions(platform)).config;
        // THEN
        expect(rendered[key]).toBeDefined();
        expect(rendered[key]).toEqual({
            httpd: {
                enabled: true,
                ensureRunning: true,
            },
        });
    });
    test.each([
        ['Linux', 'sysvinit', cfn_init_internal_1.InitPlatform.LINUX],
        ['Windows', 'windows', cfn_init_internal_1.InitPlatform.WINDOWS],
    ])('disable returns a minimalist disabled service for %s', (_platform, key, platform) => {
        // GIVEN
        const service = ec2.InitService.disable('httpd');
        // WHEN
        const rendered = service._bind(defaultOptions(platform)).config;
        // THEN
        expect(rendered[key]).toBeDefined();
        expect(rendered[key]).toEqual({
            httpd: {
                enabled: false,
                ensureRunning: false,
            },
        });
    });
    test('fromOptions renders all options', () => {
        // GIVEN
        const restartHandle = new ec2.InitServiceRestartHandle();
        const serviceRestartHandles = [restartHandle];
        const initFile = ec2.InitFile.fromString('/etc/my.cnf', '[mysql]', { serviceRestartHandles });
        const initSource = ec2.InitSource.fromUrl('/tmp/foo', 'https://example.com/archive.zip', { serviceRestartHandles });
        const initPackage = ec2.InitPackage.yum('httpd', { serviceRestartHandles });
        const initCommand = ec2.InitCommand.shellCommand('/bin/true', { serviceRestartHandles, key: 'myCmd' });
        const service = ec2.InitService.enable('httpd', { serviceRestartHandle: restartHandle });
        // WHEN
        const bindOptions = defaultOptions(cfn_init_internal_1.InitPlatform.LINUX);
        initFile._bind(bindOptions);
        initSource._bind(bindOptions);
        initPackage._bind(bindOptions);
        initCommand._bind(bindOptions);
        const rendered = service._bind(bindOptions).config;
        // THEN
        expect(rendered.sysvinit).toEqual({
            httpd: {
                enabled: true,
                ensureRunning: true,
                files: ['/etc/my.cnf'],
                sources: ['/tmp/foo'],
                packages: { yum: ['httpd'] },
                commands: ['myCmd'],
            },
        });
    });
});
describe('InitSource', () => {
    test('fromUrl renders correctly', () => {
        // GIVEN
        const source = ec2.InitSource.fromUrl('/tmp/foo', 'https://example.com/archive.zip');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://example.com/archive.zip',
        });
    });
    test('fromGitHub builds a path to the tarball', () => {
        // GIVEN
        const source = ec2.InitSource.fromGitHub('/tmp/foo', 'aws', 'aws-cdk', 'master');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://github.com/aws/aws-cdk/tarball/master',
        });
    });
    test('fromGitHub defaults to master if refspec is omitted', () => {
        // GIVEN
        const source = ec2.InitSource.fromGitHub('/tmp/foo', 'aws', 'aws-cdk');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': 'https://github.com/aws/aws-cdk/tarball/master',
        });
    });
    test('fromS3Object uses object URL', () => {
        // GIVEN
        const bucket = s3.Bucket.fromBucketName(stack, 'bucket', 'MyBucket');
        const source = ec2.InitSource.fromS3Object('/tmp/foo', bucket, 'myKey');
        // WHEN
        const rendered = getElementConfig(source, cfn_init_internal_1.InitPlatform.LINUX);
        // THEN
        expect(rendered).toEqual({
            '/tmp/foo': expect.stringContaining('/MyBucket/myKey'),
        });
    });
});
function getElementConfig(element, platform) {
    return element._bind(defaultOptions(platform)).config;
}
function defaultOptions(platform) {
    return {
        scope: stack,
        index: 0,
        platform,
        instanceRole: new iam.Role(stack, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        }),
    };
}
function createTmpFileWithContent(content) {
    const suffix = crypto.randomBytes(4).toString('hex');
    const fileName = path.join(os.tmpdir(), `cfn-init-element-test-${suffix}`);
    fs.writeFileSync(fileName, content);
    return fileName;
}
//# sourceMappingURL=data:application/json;base64,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