"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName,
            ebs,
            virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(constructs_1.Node.of(construct).uniqueId));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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