#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const glue = require("../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-cdk-glue');
const bucket = new s3.Bucket(stack, 'DataBucket');
const database = new glue.Database(stack, 'MyDatabase', {
    databaseName: 'my_database',
});
const columns = [{
        name: 'col1',
        type: glue.Schema.STRING,
    }, {
        name: 'col2',
        type: glue.Schema.STRING,
        comment: 'col2 comment',
    }, {
        name: 'col3',
        type: glue.Schema.array(glue.Schema.STRING),
    }, {
        name: 'col4',
        type: glue.Schema.map(glue.Schema.STRING, glue.Schema.STRING),
    }, {
        name: 'col5',
        type: glue.Schema.struct([{
                name: 'col1',
                type: glue.Schema.STRING,
            }]),
    }];
const partitionKeys = [{
        name: 'year',
        type: glue.Schema.SMALL_INT,
    }];
const avroTable = new glue.Table(stack, 'AVROTable', {
    database,
    bucket,
    tableName: 'avro_table',
    columns,
    partitionKeys,
    dataFormat: glue.DataFormat.AVRO,
});
const csvTable = new glue.Table(stack, 'CSVTable', {
    database,
    bucket,
    tableName: 'csv_table',
    columns,
    partitionKeys,
    dataFormat: glue.DataFormat.CSV,
});
const jsonTable = new glue.Table(stack, 'JSONTable', {
    database,
    bucket,
    tableName: 'json_table',
    columns,
    partitionKeys,
    dataFormat: glue.DataFormat.JSON,
});
const parquetTable = new glue.Table(stack, 'ParquetTable', {
    database,
    bucket,
    tableName: 'parquet_table',
    columns,
    partitionKeys,
    dataFormat: glue.DataFormat.PARQUET,
});
const encryptedTable = new glue.Table(stack, 'MyEncryptedTable', {
    database,
    tableName: 'my_encrypted_table',
    columns,
    partitionKeys: [{
            name: 'year',
            type: glue.Schema.SMALL_INT,
        }],
    dataFormat: glue.DataFormat.JSON,
    encryption: glue.TableEncryption.KMS,
    encryptionKey: new kms.Key(stack, 'MyKey'),
});
const user = new iam.User(stack, 'MyUser');
csvTable.grantReadWrite(user);
encryptedTable.grantReadWrite(user);
const anotherUser = new iam.User(stack, 'AnotherUser');
avroTable.grantReadWrite(anotherUser);
jsonTable.grantReadWrite(anotherUser);
parquetTable.grantReadWrite(anotherUser);
app.synth();
//# sourceMappingURL=data:application/json;base64,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