'''
    Helper functions for dealing with permission/data integrity issues.

    The responses generated by these functions are "api friendly" -> any js code obtaining these responses knows that an error occurred (status code 403), and can determine whether it was a permssion issue or whether the operation simply isn't allowed at this time.

    The user will NOT be redirected to the login page. There are many situations in which that doesn't work/doesn't make sense (ie. in response to xhr/api requests). Instead, the templates rendered by these helpers should provide the user the opportunity to log in.

    Our templates for these responses require that you have LOGIN_URL defined.
'''
import json

from django import http
from django.template.response import TemplateResponse

from django_early_return import EarlyReturn

import logging
log = logging.getLogger(__name__)

def assert_permission(request, permission_name):
    '''
        Ensure that the user is logged in and has the specified permission, or raise EarlyReturn with an appropriate error response.
    '''

    # TODO - if DEBUG, I think we should ensure permission_name is actually a valid permission name
    user = request.user

    if request.user.is_anonymous :
        log.info(f'Login required at {request.path_info}')
        r = TemplateResponse(  
            request,
            'qnc_crud/login_required.html',
            dict(
            ),
            status=403, 
        )
        r['ForbiddenReason'] = 'LOGIN_REQUIRED'
        raise EarlyReturn(r)

    if not user.has_perm(permission_name) :    
        log.warning(f'Permission denied at endpoint {request.path_info}: {user} (pk {user.pk}) does not have permission {permission_name}')
        r = TemplateResponse(
            request,
            'qnc_crud/permission_denied.html',
            dict(
            ),
            status=403,
        )
        # if this is in response to an ajax request, your js may want to differentiate between permission denied and "not currently allowed" type errors
        r['ForbiddenReason'] = 'PERMISSION_DENIED'
        raise EarlyReturn(r)

def not_currently_allowed(request, reason):
    '''
        Log details and raise EarlyReturn with appropriate response.
        
        Call this when user would normally have access to this data, but cannot do so right now because of app logic/data constraints.

        This is considered "less severe" than asser_permssion failures.

        Recommendation - don't bother checking for these kind of constraints in GET requests. 
        Check only before/when modifying data in a POST/PUT/DELETE request (you might even raise this in response to db.IntegrityError). 
    '''
    user = request.user
    log.info(f'Not currently allowed at {request.path_info}: {user} (pk {user.pk}) denied because {reason}')
    r = TemplateResponse(
        request,
        'qnc_crud/not_currently_allowed.html',
        dict(reason=reason),
        status=403,
    )
    # if this is in response to an ajax request, your js may want to differentiate between permission denied and "not currently allowed" type errors
    r['ForbiddenReason'] = f'NOT CURRENTLY ALLOWED: {reason}'
    raise EarlyReturn(r)