"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
* An EC2 Key Pair
*/
class KeyPair extends cdk.Construct {
    /**
    * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
    */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
        * ARN of the private key in AWS Secrets Manager
        */
        this.arn = '';
        /**
        * Name of the Key Pair
        */
        this.name = '';
        if (props.removePrivateKeyAfterDays && (props.removePrivateKeyAfterDays < 0 ||
            props.removePrivateKeyAfterDays > 0 && props.removePrivateKeyAfterDays < 7 ||
            props.removePrivateKeyAfterDays > 30)) {
            scope.node.addError(`Parameter removePrivateKeyAfterDays must be 0 or between 7 and 30. Got ${props.removePrivateKeyAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag('CreatedBy', ID);
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KeyLength: props.keyLength || KeyLength.L2048,
                Kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                RemovePrivateKeyAfterDays: props.removePrivateKeyAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-private-key/',
                StackName: stack,
                Tags: cdk.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(fn.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(fn.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ec2:CreateKeyPair',
                        'ec2:DeleteKeyPair',
                        'ec2:DescribeKeyPairs',
                        'secretsmanager:ListSecrets',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:TagResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:DeleteResourcePolicy',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:DescribeSecret',
                        'secretsmanager:GetResourcePolicy',
                        'secretsmanager:ListSecretVersionIds',
                        'secretsmanager:PutResourcePolicy',
                        'secretsmanager:PutSecretValue',
                        'secretsmanager:RestoreSecret',
                        'secretsmanager:UntagResource',
                        'secretsmanager:UpdateSecret',
                        'secretsmanager:UpdateSecretVersionStage',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    /**
    * Grants read access to the private key in AWS Secrets Manager
    */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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