import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import cdk = require('@aws-cdk/core');
export declare enum KeyLength {
    L2048 = 2048,
    L4096 = 4096
}
/**
* Definition of EC2 Key Pair
*/
export interface KeyPairProps extends cdk.ResourceProps {
    /**
    * Name of the Key Pair
    *
    * In AWS Secrets Manager the key will be prefixed with `ec2-private-key/`.
    *
    * The name can be up to 255 characters long. Valid characters include _, -, a-z, A-Z, and 0-9.
    */
    readonly name: string;
    /**
    * The description for the key in AWS Secrets Manager
    * @default - ''
    */
    readonly description?: string;
    /**
    * Number of bits in the key. Valid options are 2048 and 4096
    *
    * @default - 2048
    */
    readonly keyLength?: KeyLength;
    /**
    * The KMS key to use to encrypt the private key with
    *
    * This needs to be a key created in the same stack. You cannot use a key imported via ARN.
    *
    * @default - `alias/aws/secretsmanager`
    */
    readonly kms?: kms.Key;
    /**
    * Tags that will be applied to the private key in the AWS Secrets Manager
    *
    * EC2 Key Pairs themselves don't support tags
    *
    * @default - `alias/aws/secretsmanager`
    */
    readonly tags?: {
        [key: string]: string;
    };
    /**
    * When the resource is destroyed, after how many days the private key in the AWS Secrets Manager should be deleted.
    *
    * Valid values are 0 and 7 to 30
    *
    * @default 0
    */
    readonly removePrivateKeyAfterDays?: number;
    /**
    * Prefix for the secret in AWS Secrets Manager.
    *
    * @default `ec2-private-key/`
    */
    readonly secretPrefix?: string;
}
/**
* An EC2 Key Pair
*/
export declare class KeyPair extends cdk.Construct implements cdk.ITaggable {
    /**
    * ARN of the private key in AWS Secrets Manager
    */
    readonly arn: string;
    /**
    * Name of the Key Pair
    */
    readonly name: string;
    /**
    * Resource tags
    */
    readonly tags: cdk.TagManager;
    /**
    * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
    */
    constructor(scope: cdk.Construct, id: string, props: KeyPairProps);
    private ensureLambda;
    /**
    * Grants read access to the private key in AWS Secrets Manager
    */
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
