from typing import Any, Dict, List, Type, TypeVar, Union

import attr

from ..models.booking_field_create import BookingFieldCreate
from ..models.price_option_create import PriceOptionCreate
from ..models.product_update_request_barcode_output_type import ProductUpdateRequestBarcodeOutputType
from ..models.product_update_request_confirm_mode import ProductUpdateRequestConfirmMode
from ..types import UNSET, Unset

T = TypeVar("T", bound="ProductUpdateRequest")


@attr.s(auto_attribs=True)
class ProductUpdateRequest:
    """Partial product model containing all fields which are currently supported in product create via API.

    Attributes:
        booking_fields (List[BookingFieldCreate]): List of booking fields required for this product.
        confirm_mode (ProductUpdateRequestConfirmMode): Confirmation mode. Determines if bookings are automatically
            confirmed or it they are pending.
        description (str): Long product description, is between 100 and 15000 characters.
        short_description (str): Product description is between 15 and 240 characters.
        advertised_price (Union[Unset, float]): General price indication for this product. It represents a display price
            only, therefore it does not affect a real booking price, which is calculated based on the price options.
        barcode_output_type (Union[Unset, ProductUpdateRequestBarcodeOutputType]): Specifies how to output the barcodes
            when this product is booked. Valid types are:<br><li>PARTICIPANT: Barcodes will be generated by rezdy for each
            participant when an booking is created for this product</li><li>ORDER: Barcodes will be generated by rezdy per
            booking</li>
        confirm_mode_min_participants (Union[Unset, int]): If confirmMode is MANUAL_THEN_AUTO or AUTO_THEN_MANUAL,
            determines the minimum number of participants per booking to trigger the change.
        duration_minutes (Union[Unset, int]): Duration of the product in minutes.
        internal_code (Union[Unset, str]): Supplier-defined product code, used internally by the supplier.
        minimum_notice_minutes (Union[Unset, int]): Minimum book ahead interval for he product in minutes.
        name (Union[Unset, str]): Product name
        pickup_id (Union[Unset, int]): Pickup ID for this product.
        price_options (Union[Unset, List[PriceOptionCreate]]): List of price options for this product.
        terms (Union[Unset, str]): Specific terms and conditions for this product.
    """

    booking_fields: List[BookingFieldCreate]
    confirm_mode: ProductUpdateRequestConfirmMode
    description: str
    short_description: str
    advertised_price: Union[Unset, float] = UNSET
    barcode_output_type: Union[Unset, ProductUpdateRequestBarcodeOutputType] = UNSET
    confirm_mode_min_participants: Union[Unset, int] = UNSET
    duration_minutes: Union[Unset, int] = UNSET
    internal_code: Union[Unset, str] = UNSET
    minimum_notice_minutes: Union[Unset, int] = UNSET
    name: Union[Unset, str] = UNSET
    pickup_id: Union[Unset, int] = UNSET
    price_options: Union[Unset, List[PriceOptionCreate]] = UNSET
    terms: Union[Unset, str] = UNSET
    additional_properties: Dict[str, Any] = attr.ib(init=False, factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        booking_fields = []
        for booking_fields_item_data in self.booking_fields:
            booking_fields_item = booking_fields_item_data.to_dict()

            booking_fields.append(booking_fields_item)

        confirm_mode = self.confirm_mode.value

        description = self.description
        short_description = self.short_description
        advertised_price = self.advertised_price
        barcode_output_type: Union[Unset, str] = UNSET
        if not isinstance(self.barcode_output_type, Unset):
            barcode_output_type = self.barcode_output_type.value

        confirm_mode_min_participants = self.confirm_mode_min_participants
        duration_minutes = self.duration_minutes
        internal_code = self.internal_code
        minimum_notice_minutes = self.minimum_notice_minutes
        name = self.name
        pickup_id = self.pickup_id
        price_options: Union[Unset, List[Dict[str, Any]]] = UNSET
        if not isinstance(self.price_options, Unset):
            price_options = []
            for price_options_item_data in self.price_options:
                price_options_item = price_options_item_data.to_dict()

                price_options.append(price_options_item)

        terms = self.terms

        field_dict: Dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "bookingFields": booking_fields,
                "confirmMode": confirm_mode,
                "description": description,
                "shortDescription": short_description,
            }
        )
        if advertised_price is not UNSET:
            field_dict["advertisedPrice"] = advertised_price
        if barcode_output_type is not UNSET:
            field_dict["barcodeOutputType"] = barcode_output_type
        if confirm_mode_min_participants is not UNSET:
            field_dict["confirmModeMinParticipants"] = confirm_mode_min_participants
        if duration_minutes is not UNSET:
            field_dict["durationMinutes"] = duration_minutes
        if internal_code is not UNSET:
            field_dict["internalCode"] = internal_code
        if minimum_notice_minutes is not UNSET:
            field_dict["minimumNoticeMinutes"] = minimum_notice_minutes
        if name is not UNSET:
            field_dict["name"] = name
        if pickup_id is not UNSET:
            field_dict["pickupId"] = pickup_id
        if price_options is not UNSET:
            field_dict["priceOptions"] = price_options
        if terms is not UNSET:
            field_dict["terms"] = terms

        return field_dict

    @classmethod
    def from_dict(cls: Type[T], src_dict: Dict[str, Any]) -> T:
        d = src_dict.copy()
        booking_fields = []
        _booking_fields = d.pop("bookingFields")
        for booking_fields_item_data in _booking_fields:
            booking_fields_item = BookingFieldCreate.from_dict(booking_fields_item_data)

            booking_fields.append(booking_fields_item)

        confirm_mode = ProductUpdateRequestConfirmMode(d.pop("confirmMode"))

        description = d.pop("description")

        short_description = d.pop("shortDescription")

        advertised_price = d.pop("advertisedPrice", UNSET)

        _barcode_output_type = d.pop("barcodeOutputType", UNSET)
        barcode_output_type: Union[Unset, ProductUpdateRequestBarcodeOutputType]
        if isinstance(_barcode_output_type, Unset):
            barcode_output_type = UNSET
        else:
            barcode_output_type = ProductUpdateRequestBarcodeOutputType(_barcode_output_type)

        confirm_mode_min_participants = d.pop("confirmModeMinParticipants", UNSET)

        duration_minutes = d.pop("durationMinutes", UNSET)

        internal_code = d.pop("internalCode", UNSET)

        minimum_notice_minutes = d.pop("minimumNoticeMinutes", UNSET)

        name = d.pop("name", UNSET)

        pickup_id = d.pop("pickupId", UNSET)

        price_options = []
        _price_options = d.pop("priceOptions", UNSET)
        for price_options_item_data in _price_options or []:
            price_options_item = PriceOptionCreate.from_dict(price_options_item_data)

            price_options.append(price_options_item)

        terms = d.pop("terms", UNSET)

        product_update_request = cls(
            booking_fields=booking_fields,
            confirm_mode=confirm_mode,
            description=description,
            short_description=short_description,
            advertised_price=advertised_price,
            barcode_output_type=barcode_output_type,
            confirm_mode_min_participants=confirm_mode_min_participants,
            duration_minutes=duration_minutes,
            internal_code=internal_code,
            minimum_notice_minutes=minimum_notice_minutes,
            name=name,
            pickup_id=pickup_id,
            price_options=price_options,
            terms=terms,
        )

        product_update_request.additional_properties = d
        return product_update_request

    @property
    def additional_keys(self) -> List[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
