"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks_generated_1 = require("./eks.generated");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.nodegroupName
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com')
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: props.amiType,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_e = props.forceUpdate) !== null && _e !== void 0 ? _e : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize
            },
            tags: props.tags
        });
        // As managed nodegroup will auto map the instance role to RBAC behind the scene and users don't have to manually
        // do it anymore. We don't need to print out the instance role arn now.
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
//# sourceMappingURL=data:application/json;base64,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