"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_patch_1 = require("./k8s-patch");
const k8s_resource_1 = require("./k8s-resource");
const managed_nodegroup_1 = require("./managed-nodegroup");
const spot_interrupt_handler_1 = require("./spot-interrupt-handler");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSServicePolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds
            }
        };
        let resource;
        this.kubectlEnabled = props.kubectlEnabled === undefined ? true : props.kubectlEnabled;
        if (this.kubectlEnabled) {
            resource = new cluster_resource_1.ClusterResource(this, 'Resource', clusterProps);
            this._clusterResource = resource;
        }
        else {
            resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        }
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // map the IAM role to the `system:masters` group.
        if (props.mastersRole) {
            if (!this.kubectlEnabled) {
                throw new Error('Cannot specify a "masters" role if kubectl is disabled');
            }
            this.awsAuth.addMastersRole(props.mastersRole);
            if (props.outputMastersRoleArn) {
                new core_1.CfnOutput(this, 'MastersRoleArn', { value: props.mastersRole.roleArn });
            }
            commonCommandOptions.push(`--role-arn ${props.mastersRole.roleArn}`);
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        if (this.kubectlEnabled) {
            this.defineCoreDnsComputeType(props.coreDnsComputeType || CoreDnsComputeType.EC2);
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: new EksOptimizedImage({
                nodeType: nodeTypeForInstanceType(options.instanceType),
                kubernetesVersion: this.version,
            }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled
        });
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        var _a;
        // initialize the awsAuth for this cluster
        this._awsAuth = (_a = this._awsAuth) !== null && _a !== void 0 ? _a : this.awsAuth;
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = user_data_1.renderUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tag.add(autoScalingGroup, `kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true
        });
        if (options.mapRole === true && !this.kubectlEnabled) {
            throw new Error('Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster');
        }
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole && this.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes'
                ]
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && this.kubectlEnabled) {
            this.addResource('spot-interrupt-handler', ...spot_interrupt_handler_1.spotInterruptHandler());
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot define aws-auth mappings if kubectl is disabled');
        }
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    _getKubectlCreationRoleArn(assumedBy) {
        if (!this._clusterResource) {
            throw new Error('Unable to perform this operation since kubectl is not enabled for this cluster');
        }
        return this._clusterResource.getCreationRoleArn(assumedBy);
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                subnet.node.applyAspect(new core_1.Tag(tag, '1'));
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        if (!this.kubectlEnabled) {
            throw new Error('kubectl must be enabled in order to define the compute type for CoreDNS');
        }
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType
                        }
                    }
                }
            }
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2)
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.nodeType = props && props.nodeType;
        this.kubernetesVersion = props && props.kubernetesVersion || LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU : NodeType.STANDARD;
}
exports.nodeTypeForInstanceType = nodeTypeForInstanceType;
//# sourceMappingURL=data:application/json;base64,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