# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_tensors.ipynb.

# %% auto 0
__all__ = ['PRINT_OPTS', 'pretty_str', 'lovely', 'rgb', 'plot', 'monkey_patch']

# %% ../nbs/00_tensors.ipynb 2
from typing import Optional, Union
import math

from nbdev.showdoc import *
from PIL import Image

from matplotlib import pyplot as plt
from IPython.core.pylabtools import print_figure

import torch

from fastcore.test import test, test_eq, is_close
from fastcore.foundation import patch_to

# import wandb

# %% ../nbs/00_tensors.ipynb 3
class __PrinterOptions(object):
    precision: int = 3
    threshold_max: int = 3 # .abs() larger than 1e3 -> Sci mode
    threshold_min: int = -4 # .abs() smaller that 1e-4 -> Sci mode
    sci_mode: Optional[bool] = None # None = auto. Otherwise, force sci mode.
    indent: int = 2 # Indent for .deeper()
    color: bool = True


# %% ../nbs/00_tensors.ipynb 4
PRINT_OPTS = __PrinterOptions()

# %% ../nbs/00_tensors.ipynb 5
# Do we want this float in decimal or scientific mode?
def sci_mode(f: float):
    return (abs(f) < 10**(PRINT_OPTS.threshold_min) or
            abs(f) > 10**PRINT_OPTS.threshold_max)

# %% ../nbs/00_tensors.ipynb 8
# Convert a tensor into a string.
# This only looks good for small tensors, which is how it's intended to be used.
def pretty_str(t: Union[torch.Tensor, float, int]):
    """A slightly better way to print `floaty` values"""

    if isinstance(t, int):
        return '{}'.format(t)
    elif isinstance(t, float):
        if t == 0.:
            return "0."

        sci = (PRINT_OPTS.sci_mode or
                (PRINT_OPTS.sci_mode is None and sci_mode(t)))
        # The f-string will generate something like "{.4f}", which is used
        # to format the value.
        return f"{{:.{PRINT_OPTS.precision}{'e' if sci else 'f'}}}".format(t)
    elif t.dim() == 0:
            return pretty_str(t.item())
    else:
        slices = [pretty_str(t[i]) for i in range(0, t.size(0))]
        return '[' + ", ".join(slices) + ']'

# %% ../nbs/00_tensors.ipynb 13
def space_join(lst):
    # Join non-empty list elements into a space-sepaeated string
    return " ".join( [ l for l in lst if l] )

# %% ../nbs/00_tensors.ipynb 15
class LovelyProxy():
    def __init__(self, t: torch.Tensor, plain=False, verbose=False, depth=0, lvl=0, color=None):
        self.t = t
        self.plain = plain
        self.verbose = verbose
        self.depth=depth
        self.lvl=lvl
        self.color=color

    @torch.no_grad()
    def to_str(self):
        t = self.t
        if self.plain:
            return torch._tensor_str._tensor_str(t, indent=0)

        color = PRINT_OPTS.color if self.color is None else self.color
        
        grey_style = "\x1b[38;2;127;127;127m" if color else ""
        red_style = "\x1b[31m" if color else ""
        end_style = "\x1b[0m" if color else ""

        dtnames = { torch.float32: "",
                    torch.float16: "f16",
                    torch.float64: "f64",
                    torch.uint8: "u8",
                    torch.int32: "i32",
                }


        tname = "tensor" if type(t) is torch.Tensor else type(t).__name__
        dev = str(t.device) if t.device.type != "cpu" else None
        dtype = dtnames[t.dtype] if t.dtype in dtnames else str(t.dtype)[6:]


        grad_fn = "grad_fn" if t.grad_fn else None
        # All tensors along the compute path actually have required_grad=True.
        # Torch __repr__ just dones not show it.
        grad = "grad" if t.requires_grad and not t.grad_fn else None

        shape = str(list(t.shape))

        # Later, we might be indexing 't' with a bool tensor derived from it. 
        # THis takes 4x memory and will result in a CUDA OOM if 't' is very large.
        # Move it to the cpu now - it won't matter for small tensors, and for
        # very large ones we trade a CUDA OOM for a few seconds delay.
        t = t.cpu()

        zeros = grey_style+"all_zeros"+end_style if t.eq(0.).all() and t.numel() > 1 else None
        pinf = red_style+"+inf!"+end_style if t.isposinf().any() else None
        ninf = red_style+"-inf!"+end_style if t.isneginf().any() else None
        nan = red_style+"nan!"+end_style if t.isnan().any() else None

        # zeros = "all_zeros" if not t.count_nonzero() else None
        # pinf = "+inf!" if t.isposinf().any() else None
        # ninf = "-inf!" if t.isneginf().any() else None
        # nan = "nan!" if t.isnan().any() else None

        attention = space_join([zeros,pinf,ninf,nan])

        x = ""
        summary = f"n={t.numel()}" if t.numel() > 5 else None
        if not zeros:
            if t.numel() <= 10: x = pretty_str(t)
            
            # Make sure it's float32. Also, we calculate stats on good values only.

            ft = t[ torch.isfinite(t) ].float()

            minmax = f"x∈[{pretty_str(ft.min())}, {pretty_str(ft.max())}]" if t.numel() > 2 and ft.numel() > 2 else None
            meanstd = f"μ={pretty_str(ft.mean())} σ={pretty_str(ft.std())}" if t.numel() >= 2 and ft.numel() >= 2 else None
            numel = f"n={t.numel()}" if t.numel() > 5 and max(t.shape) != t.numel() else None

            summary = space_join([numel, minmax, meanstd])




        res = tname + space_join([  shape,
                                    summary,
                                    dtype,
                                    grad,
                                    grad_fn,
                                    dev,
                                    attention,
                                    x if not self.verbose else None])

        if self.verbose:
            res += "\n" + torch._tensor_str._tensor_str(t, indent=0)

        if self.depth and t.dim() > 1:
            res += "\n"
            # for i in range(t.shape[0]):
                # str = 
            res += "\n".join([
                " "*PRINT_OPTS.indent*(self.lvl+1) +
                str(LovelyProxy(t[i,:], depth=self.depth-1, lvl=self.lvl+1))
                for i in range(t.shape[0])])

        return res
    
    def __repr__(self):
        return self.to_str()

    def __call__(self, depth=0):
        return LovelyProxy(self.t, depth=depth)

# %% ../nbs/00_tensors.ipynb 17
def lovely(t: torch.Tensor, verbose=False, plain=False, depth=0, color=None):
    return LovelyProxy(t, verbose=verbose, plain=plain, depth=depth, color=color)

# %% ../nbs/00_tensors.ipynb 32
# This is here for the monkey-patched tensor use case.

# I want to be able to call both `tensor.rgb` and `tensor.rgb(stats)`. For the
# first case, the class defines `_repr_png_` to send the image to Jupyter. For
# the later case, it defines __call__, which accps the argument.
class ProxyImage(Image.Image): 
    """Flexible `PIL.Image.Image` wrapper"""
    @torch.no_grad()
    def __init__(self, t:torch.Tensor):
        super().__init__()
        self.t = t.permute(1, 2, 0)

        # Mode and size - to be used by super().__repr__()
        self.mode = "RGB"
        self._size = tuple(t.shape[1:])

    @torch.no_grad()
    def __call__(self, denorm=None):            
        means = torch.tensor(denorm["mean"] if denorm else (0., 0., 0.,), device=self.t.device)
        stds = torch.tensor(denorm["std"] if denorm else (1., 1., 1.,), device=self.t.device)
        return Image.fromarray((self.t*stds+means).mul(255).byte().cpu().numpy())

    @torch.no_grad()
    def _repr_png_(self):
        "Jupyter PNG representation"
        return Image.fromarray(self.t.mul(255).byte().cpu().numpy())._repr_png_()


# %% ../nbs/00_tensors.ipynb 33
def rgb(t: torch.Tensor, denorm=None):
    return ProxyImage(t)(denorm)

# %% ../nbs/00_tensors.ipynb 36
def normal_pdf( x: torch.Tensor,
                mean: Union[torch.Tensor, float] =0.,
                std: Union[torch.Tensor, float] =1.):
    r"""Probability Distribution Function of Normal Distribution:
        $$f(x, \mu, \sigma)
        = \dfrac{1}
            {\sigma \sqrt{2\pi} }
        e^{-\frac{1}{2} \left(\frac{x-\mu}{\sigma}\right)^{2}}
        = \dfrac{e^{- \frac{1}{2}\left(\frac{x-\mu}{\sigma}\right)^{2}}}
            {\sigma \sqrt{2\pi}}$$"""

    dev = x.device
    e = torch.tensor(math.e).to(dev)
    pi = torch.tensor(math.pi).to(dev)

    mean = torch.tensor(mean).to(dev) if not isinstance(mean, torch.Tensor) else mean
    std = torch.tensor(std).to(dev) if not isinstance(std, torch.Tensor) else std

    return (e
                .pow( -0.5 * ((x - mean)/std).pow(2) )
                .div((std * (pi * 2).sqrt()))
            )


# %% ../nbs/00_tensors.ipynb 38
@torch.no_grad()
def plot(t: torch.Tensor, ax=None, center="zero", limit=100000):
    """Plot tensor statistics"""

    assert center in ["zero", "mean", "range"]

    t_str = str(lovely(t, color=False))

    # Same as in `lovely()`, we have to move it to cpu before good-value indexing.
    t = t.cpu()

    # `t`` may have nasty things like 'nan' and 'inf'. Could also be of non-float type.
    t = t[ torch.isfinite(t) ].float()

    t_min, t_max = t.min().item(), t.max().item()
    t_mean, t_std = t.mean().item(), t.std().item()
    
    # `t` might be not on CPU. `t` also might be pretty large. If the tensor is large,
    # randomly sample up to `limit` before moving it to cpu. Here sampling does
    # not cause excessice CUDA memory allocation, because the index tensor is not large.   
    if t.numel() > limit:
        # For efficieny reasons, we have to sample with replacement.
        idxs = torch.randint(low=0, high=t.numel(), size=(limit,), device=t.device)
        t = t.flatten()[idxs]
        t_str = f"{limit} samples of " + t_str

    if t.numel() < 10:
        return

    fig = None
    if not ax:
        fig, ax = plt.subplots(figsize=(12, 2), layout="tight")
        plt.close(fig)

    # Center the plot around zero, mean, or the extents of the range of t.
    if center == "range":
        # X limits should center the plot around the mean value
        # x_limit = max(t_min.abs().item(), t_max.abs().item())
        x_min, x_max = t_min, t_max
    elif center == "mean":
        max_div = max((t_mean - abs(t_min)), (abs(t_max) - t_mean))
        x_min, x_max = t_mean - max_div, t_mean + max_div
        # x_limits = [t_mean - x_limit, t_mean + x_limit]
    else:
        # X limits should center the plot around zero
        abs_max_value = max(abs(t_min), abs(t_max))
        # ,
                    # (t_mean-max_sigma*t_std).item(), (t_mean+max_sigma*t_std).item())
        x_min, x_max = -abs_max_value, abs_max_value

    sigmas = max(int(math.floor((abs(t_mean - t_min) / t_std))),
        int(math.floor((abs(t_max - t_mean) / t_std))))

    x_min -= abs(x_max - x_min) * 0.02
    x_max += abs(x_max - x_min) * 0.02

    # Around 50 items / bin seems ot look good. But don't go below 10 or above 100.
    bins = int(t.numel() / 50)
    bins = max(min(bins, 100), 10)

    histc = t.histc(bins=bins, min=t_min, max=t_max)

    bar_edges = torch.linspace(t_min, t_max, bins+1)[:bins]
    bar_width = bar_edges[:2].diff()

    # Histogram normalized to look like PDF: area under histogram = 1.
    histc_density = (histc / (histc.sum() * bar_width))
    ax.bar(x=bar_edges.numpy(), height=histc_density.numpy(), width=bar_width, color="deepskyblue", align="edge", zorder=4,)

    # PDF of normal distribution with the same mean and std.
    x = torch.linspace(x_min, x_max, 100)
    normal_density = normal_pdf(x, mean=t_mean, std=t_std)
    ax.plot(x, normal_pdf(x, mean=t_mean, std=t_std), zorder=5)

    y_lim = max(histc_density.max().item(), normal_density.max().item()) * 1.3

    # Make text bank part of the line under it
    bbox = dict(boxstyle="round", fc="white", edgecolor="none")

    for s in range(-sigmas, sigmas+1):
        x_pos = (t_mean + s*t_std)
        if x_min < x_pos < x_max:
            greek = ["-σ", "μ", "+σ"][s+1]  if -1 <= s <= 1 else f"{s:+}σ"
            weight='bold' if not s else None
            ax.axvline(x_pos, 0, 1, c="black")
            ax.text(x_pos, y_lim*0.95, greek, ha="center", va="top", bbox=bbox, zorder=5, weight=weight)

    # 2 red lines for min and max values
    ax.annotate(
        f"min={pretty_str(t_min)}",
        (t_min, y_lim/2),
        xytext=(-1, 0), textcoords='offset points',
        bbox=bbox,
        rotation=90,
        ha="right",
        va="center"
        )

    ax.annotate(
        f"max={pretty_str(t_max)}",
        (t_max, y_lim/2),
        xytext=(2, 0), textcoords='offset points',
        bbox=bbox,
        rotation=90,
        ha="left",
        va="center"
        )

    ax.axvline(t_min, 0, 1, c="red", zorder=4)
    ax.axvline(t_max, 0, 1, c="red", zorder=4)

    ax.text(x_min, y_lim*1.05, s=t_str)
    ax.set_ylim(0, y_lim)
    ax.set_yticks([])

    ax.set_xlim(x_min, x_max )

    return fig

# %% ../nbs/00_tensors.ipynb 40
# This is here for the monkey-patched tensor use case.
# Gives the ability to call both .plt and .plt(ax=ax).  

class ProxyPlot(): 
    """Flexible `PIL.Image.Image` wrapper"""
    @torch.no_grad()
    def __init__(self, t:torch.Tensor, ax=None, center="zero", fmt="svg"):
        self.t = t # Avoid a memory leak - python seems to 
        self.ax = ax
        self.center = center
        self.fmt = fmt
        
        assert fmt in ["png", "svg"]
        assert center in ["zero", "mean", "range"]

    def __call__(self, ax=None, center="zero", fmt="svg"):
        plot = ProxyPlot(self.t, ax=ax, center=center)
        # Python seems to defer collecting self for a long time.
        # If 't' is very large, repeated calls to .plt will exhaust CUDA memory.
        # This trick seems to do the job.
        del self.t 
        return plot

    # Do an explicit print_figure instead of relying on IPythons repr formatter
    # for pyplot.Figure. Mainly for speed.
    #
    # IPython will attempt to render the figure in a bunch of formats, and then
    # pick one to show. This takes a noticeable amount of time. Render just
    # one format instead.
    def _repr_svg_(self):
        if self.fmt == "svg":
            fig = print_figure(plot(self.t, self.ax, self.center), fmt="svg")
            del self.t
            return fig

    def _repr_png_(self):
        if self.fmt == "png":
            fig = print_figure(plot(self.t, self.ax, self.center), fmt="png")
            del self.t
            return fig

# %% ../nbs/00_tensors.ipynb 41
def monkey_patch(cls=torch.Tensor):
    "Monkey-patch lovely features into `cls`" 

    @patch_to(cls)
    def __repr__(self: torch.Tensor, *, tensor_contents=None):        
        return str(LovelyProxy(self))

    # Keep an esiy way to get the standard behavior.
    @patch_to(cls, as_prop=True)
    def plain(self: torch.Tensor, *, tensor_contents=None):
        return LovelyProxy(self, plain=True)

    # And a verbose option for a good measure.
    @patch_to(cls, as_prop=True)
    def verbose(self: torch.Tensor, *, tensor_contents=None):
        return LovelyProxy(self, verbose=True)

    @patch_to(cls, as_prop=True)
    def deeper(self: torch.Tensor):
        return LovelyProxy(self, depth=1)

    @patch_to(cls, as_prop=True)
    def rgb(t: torch.Tensor):
        return ProxyImage(t)

    @patch_to(cls, as_prop=True)
    def plt(t: torch.Tensor):
        return ProxyPlot(t)
