from __future__ import annotations

from typing import Any

from plain.assets.urls import get_asset_url
from plain.auth.views import AuthViewMixin
from plain.forms import Form
from plain.http import Response, ResponseRedirect
from plain.runtime import settings
from plain.utils.module_loading import import_string
from plain.views import FormView, View


class SupportFormView(AuthViewMixin, FormView):
    template_name = "support/page.html"

    def get_form(self) -> Form:
        form_slug = self.url_kwargs["form_slug"]
        form_class = import_string(settings.SUPPORT_FORMS[form_slug])
        return form_class(**self.get_form_kwargs())

    def get_template_context(self) -> dict[str, Any]:
        context = super().get_template_context()
        form_slug = self.url_kwargs["form_slug"]
        context["form_action"] = self.request.build_absolute_uri()
        context["form_template_name"] = f"support/forms/{form_slug}.html"
        context["success_template_name"] = f"support/success/{form_slug}.html"
        context["success"] = self.request.query_params.get("success") == "true"
        return context

    def get_form_kwargs(self) -> dict[str, Any]:
        kwargs = super().get_form_kwargs()
        kwargs["user"] = self.user
        kwargs["form_slug"] = self.url_kwargs["form_slug"]
        return kwargs

    def form_valid(self, form: Any) -> Response:
        entry = form.save()
        form.notify(entry)
        return super().form_valid(form)

    def get_success_url(self, form: Any) -> str:
        # Redirect to the same view and template so we
        # don't have to create two additional views for iframe and non-iframe.
        return "?success=true"


class SupportIFrameView(SupportFormView):
    template_name = "support/iframe.html"

    def get_response(self) -> Response:
        response = super().get_response()

        # X-Frame-Options are typically in DEFAULT_RESPONSE_HEADERS.
        # Set to None to signal the middleware to skip applying this default header.
        # We can't del/pop it because middleware runs after and would add it back.
        response.headers["X-Frame-Options"] = None

        return response


class SupportFormJSView(View):
    def get(self) -> ResponseRedirect:
        return ResponseRedirect(get_asset_url("support/embed.js"))
