"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildAction = exports.CodeBuildActionType = void 0;
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const __1 = require("..");
const action_1 = require("../action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category - Build, or Test.
 *
 * The default is Build.
 *
 * @stability stable
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 *
 * @stability stable
 */
class CodeBuildAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     *
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     * @stability stable
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, _stage, options) {
        var _a;
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                `codebuild:${this.props.executeBatchBuild ? 'StartBuildBatch' : 'StartBuild'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StopBuildBatch' : 'StopBuild'}`,
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        // if any of the inputs come from the BitBucketSourceAction
        // with codeBuildCloneOutput=true,
        // grant the Project's Role to use the connection
        for (const inputArtifact of this.actionProperties.inputs || []) {
            const connectionArn = inputArtifact.getMetadata(__1.BitBucketSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables, (_a = this.props.checkSecretsInPlainTextEnvVariables) !== null && _a !== void 0 ? _a : true)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.string({ produce: () => this.props.input.artifactName });
        }
        if (this.props.executeBatchBuild) {
            configuration.BatchEnabled = 'true';
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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