#!/usr/bin/env python3
# -*- coding:utf-8 -*-
#
# This module provides a framework for the use of DNS Service Discovery
# using IP multicast.
#
# Copyright (c) 2017 François Wautier
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies
# or substantial portions of the Software.
#
# Portion of this code is covered by the following license
#
# Copyright 2003 Paul Scott-Murphy, 2014 William McBrine
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
# IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301
# USA


import asyncio
import enum
import errno
import logging
import re
import socket
import struct
import sys
import time

from abc import abstractmethod
from functools import partial, reduce

import netifaces

from typing import List, Union

__author__ = 'François Wautier'
__maintainer__ = 'François Wautier <francois@wautier.eu>'
__version__ = '0.1.0'
__license__ = 'GPL'

__all__ = [
    "__version__",
    "Zeroconf", "ServiceInfo", "ServiceBrowser", "ZeroconfServiceTypes",
    "MDNSError", "InterfaceChoice", "ServiceStateChange",
]

log = logging.getLogger(__name__)
log.addHandler(logging.NullHandler())

if log.level == logging.NOTSET:
    log.setLevel(logging.WARN)
# Some timing constants

_UNREGISTER_TIME = 125
_CHECK_TIME = 175
_REGISTER_TIME = 225
_LISTENER_TIME = 200
_BROWSER_TIME = 500

# Some DNS constants

_MDNS_ADDR = '224.0.0.251'
_MDNS6_ADDR = 'FF02::FB'
_MDNS_PORT = 5353
_DNS_PORT = 53
_DNS_TTL = 60 * 60  # one hour default TTL

_MAX_MSG_TYPICAL = 1460  # unused
_MAX_MSG_ABSOLUTE = 8966

_FLAGS_QR_MASK = 0x8000  # query response mask
_FLAGS_QR_QUERY = 0x0000  # query
_FLAGS_QR_RESPONSE = 0x8000  # response

_FLAGS_AA = 0x0400  # Authoritative answer
_FLAGS_TC = 0x0200  # Truncated
_FLAGS_RD = 0x0100  # Recursion desired
_FLAGS_RA = 0x8000  # Recursion available

_FLAGS_Z = 0x0040  # Zero
_FLAGS_AD = 0x0020  # Authentic data
_FLAGS_CD = 0x0010  # Checking disabled

_CLASS_IN = 1
_CLASS_CS = 2
_CLASS_CH = 3
_CLASS_HS = 4
_CLASS_NONE = 254
_CLASS_ANY = 255
_CLASS_MASK = 0x7FFF
_CLASS_UNIQUE = 0x8000

_TYPE_A = 1
_TYPE_NS = 2
_TYPE_MD = 3
_TYPE_MF = 4
_TYPE_CNAME = 5
_TYPE_SOA = 6
_TYPE_MB = 7
_TYPE_MG = 8
_TYPE_MR = 9
_TYPE_NULL = 10
_TYPE_WKS = 11
_TYPE_PTR = 12
_TYPE_HINFO = 13
_TYPE_MINFO = 14
_TYPE_MX = 15
_TYPE_TXT = 16
_TYPE_AAAA = 28
_TYPE_SRV = 33
_TYPE_ANY = 255

# Mapping constants to names

_CLASSES = {_CLASS_IN: "in",
            _CLASS_CS: "cs",
            _CLASS_CH: "ch",
            _CLASS_HS: "hs",
            _CLASS_NONE: "none",
            _CLASS_ANY: "any"}

_TYPES = {_TYPE_A: "a",
          _TYPE_NS: "ns",
          _TYPE_MD: "md",
          _TYPE_MF: "mf",
          _TYPE_CNAME: "cname",
          _TYPE_SOA: "soa",
          _TYPE_MB: "mb",
          _TYPE_MG: "mg",
          _TYPE_MR: "mr",
          _TYPE_NULL: "null",
          _TYPE_WKS: "wks",
          _TYPE_PTR: "ptr",
          _TYPE_HINFO: "hinfo",
          _TYPE_MINFO: "minfo",
          _TYPE_MX: "mx",
          _TYPE_TXT: "txt",
          _TYPE_AAAA: "quada",
          _TYPE_SRV: "srv",
          _TYPE_ANY: "any"}

_HAS_A_TO_Z = re.compile(r'[A-Za-z]')
_HAS_ONLY_A_TO_Z_NUM_HYPHEN = re.compile(r'^[A-Za-z0-9\-]+$')
_HAS_ASCII_CONTROL_CHARS = re.compile(r'[\x00-\x1f\x7f]')


@enum.unique
class InterfaceChoice(enum.Enum):
    Default = 1
    All = 2


@enum.unique
class ServiceStateChange(enum.Enum):
    Added = 1
    Removed = 2


HOST_ONLY_NETWORK_MASK = '255.255.255.255'


# utility functions


def current_time_millis():
    """Current system time in milliseconds"""
    return time.time() * 1000


def service_type_name(type_):
    """
    Validate a fully qualified service name, instance or subtype. [rfc6763]

    Returns fully qualified service name.

    Domain names used by mDNS-SD take the following forms:

                   <sn> . <_tcp|_udp> . local.
      <Instance> . <sn> . <_tcp|_udp> . local.
      <sub>._sub . <sn> . <_tcp|_udp> . local.

    1) must end with 'local.'

      This is true because we are implementing mDNS and since the 'm' means
      multi-cast, the 'local.' domain is mandatory.

    2) local is preceded with either '_udp.' or '_tcp.'

    3) service name <sn> precedes <_tcp|_udp>

      The rules for Service Names [RFC6335] state that they may be no more
      than fifteen characters long (not counting the mandatory underscore),
      consisting of only letters, digits, and hyphens, must begin and end
      with a letter or digit, must not contain consecutive hyphens, and
      must contain at least one letter.

    The instance name <Instance> and sub type <sub> may be up to 63 bytes.

    The portion of the Service Instance Name is a user-
    friendly name consisting of arbitrary Net-Unicode text [RFC5198]. It
    MUST NOT contain ASCII control characters (byte values 0x00-0x1F and
    0x7F) [RFC20] but otherwise is allowed to contain any characters,
    without restriction, including spaces, uppercase, lowercase,
    punctuation -- including dots -- accented characters, non-Roman text,
    and anything else that may be represented using Net-Unicode.

    :param type_: Type, SubType or service name to validate
    :return: fully qualified service name (eg: _http._tcp.local.)
    """
    if not (type_.endswith('._tcp.local.') or type_.endswith('._udp.local.')):
        raise BadTypeInNameException(
            "Type '%s' must end with '._tcp.local.' or '._udp.local.'" %
            type_)

    remaining = type_[:-len('._tcp.local.')].split('.')
    name = remaining.pop()
    if not name:
        raise BadTypeInNameException("No Service name found")

    if len(remaining) == 1 and len(remaining[0]) == 0:
        raise BadTypeInNameException(
            "Type '%s' must not start with '.'" % type_)

    if name[0] != '_':
        raise BadTypeInNameException(
            "Service name (%s) must start with '_'" % name)

    # remove leading underscore
    name = name[1:]

    if len(name) > 15:
        raise BadTypeInNameException(
            "Service name (%s) must be <= 15 bytes" % name)

    if '--' in name:
        raise BadTypeInNameException(
            "Service name (%s) must not contain '--'" % name)

    if '-' in (name[0], name[-1]):
        raise BadTypeInNameException(
            "Service name (%s) may not start or end with '-'" % name)

    if not _HAS_A_TO_Z.search(name):
        raise BadTypeInNameException(
            "Service name (%s) must contain at least one letter (eg: 'A-Z')" %
            name)

    if not _HAS_ONLY_A_TO_Z_NUM_HYPHEN.search(name):
        raise BadTypeInNameException(
            "Service name (%s) must contain only these characters: "
            "A-Z, a-z, 0-9, hyphen ('-')" % name)

    if remaining and remaining[-1] == '_sub':
        remaining.pop()
        if len(remaining) == 0 or len(remaining[0]) == 0:
            raise BadTypeInNameException(
                "_sub requires a subtype name")

    if len(remaining) > 1:
        remaining = ['.'.join(remaining)]

    if remaining:
        length = len(remaining[0].encode('utf-8'))
        if length > 63:
            raise BadTypeInNameException("Too long: '%s'" % remaining[0])

        if _HAS_ASCII_CONTROL_CHARS.search(remaining[0]):
            raise BadTypeInNameException(
                "Ascii control character 0x00-0x1F and 0x7F illegal in '%s'" %
                remaining[0])

    return '_' + name + type_[-len('._tcp.local.'):]


# Exceptions


class MDNSError(Exception):
    pass


class IncomingDecodeError(MDNSError):
    pass


class NonUniqueNameException(MDNSError):
    pass


class NamePartTooLongException(MDNSError):
    pass


class BadTypeInNameException(MDNSError):
    pass


# implementation classes


class QuietLogger(object):
    _seen_logs = {}

    @classmethod
    def log_exception_warning(cls, logger_data=None):
        exc_info = sys.exc_info()
        exc_str = str(exc_info[1])
        if exc_str not in cls._seen_logs:
            # log at warning level the first time this is seen
            cls._seen_logs[exc_str] = exc_info
            logger = log.warning
        else:
            logger = log.debug
        if logger_data is not None:
            logger(*logger_data)
        logger('Exception occurred:', exc_info=exc_info)

    @classmethod
    def log_warning_once(cls, *args):
        msg_str = args[0]
        if msg_str not in cls._seen_logs:
            cls._seen_logs[msg_str] = 0
            logger = log.warning
        else:
            logger = log.debug
        cls._seen_logs[msg_str] += 1
        logger(*args)


class DNSEntry(object):
    """A DNS entry"""

    def __init__(self, name, type_, class_):
        self.key = name.lower()
        self.name = name
        self.type = type_
        self.class_ = class_ & _CLASS_MASK
        self.unique = (class_ & _CLASS_UNIQUE) != 0

    def __eq__(self, other):
        """Equality test on name, type, and class"""
        return (isinstance(other, DNSEntry) and
                self.name == other.name and
                self.type == other.type and
                self.class_ == other.class_)

    def __ne__(self, other):
        """Non-equality test"""
        return not self.__eq__(other)

    @staticmethod
    def get_class_(class_):
        """Class accessor"""
        return _CLASSES.get(class_, "?(%s)" % class_)

    @staticmethod
    def get_type(t):
        """Type accessor"""
        return _TYPES.get(t, "?(%s)" % t)

    def to_string(self, hdr, other):
        """String representation with additional information"""
        result = "%s[%s,%s" % (hdr, self.get_type(self.type),
                               str(self.get_class_(self.class_)))
        if self.unique:
            result += "-unique,"
        else:
            result += ","
        result += self.name
        if other is not None:
            result += ",%s]" % other
        else:
            result += "]"
        return result


class DNSQuestion(DNSEntry):
    """A DNS question entry"""

    def __init__(self, name, type_, class_):
        DNSEntry.__init__(self, name, type_, class_)

    def answered_by(self, rec):
        """Returns true if the question is answered by the record"""
        return (self.class_ == rec.class_ and
                (self.type == rec.type or self.type == _TYPE_ANY) and
                self.name == rec.name)

    def __repr__(self):
        """String representation"""
        return DNSEntry.to_string(self, "question", None)


class DNSRecord(DNSEntry):
    """A DNS record - like a DNS entry, but has a TTL"""

    def __init__(self, name, type_, class_, ttl):
        DNSEntry.__init__(self, name, type_, class_)
        self.ttl = ttl
        self.created = current_time_millis()

    @abstractmethod
    def __eq__(self, other):
        """All records must implement this"""
        raise NotImplementedError

    def suppressed_by(self, msg):
        """Returns true if any answer in a message can suffice for the
        information held in this record."""
        for record in msg.answers:
            if self.suppressed_by_answer(record):
                return True
        return False

    def suppressed_by_answer(self, other):
        """Returns true if another record has same name, type and class,
        and if its TTL is at least half of this record's."""
        return self == other and other.ttl > (self.ttl / 2)

    def get_expiration_time(self, percent):
        """Returns the time at which this record will have expired
        by a certain percentage."""
        return self.created + (percent * self.ttl * 10)

    def get_remaining_ttl(self, now):
        """Returns the remaining TTL in seconds."""
        return max(0, (self.get_expiration_time(100) - now) / 1000.0)

    def is_expired(self, now):
        """Returns true if this record has expired."""
        return self.get_expiration_time(100) <= now

    def is_stale(self, now):
        """Returns true if this record is at least half way expired."""
        return self.get_expiration_time(50) <= now

    def reset_ttl(self, other):
        """Sets this record's TTL and created time to that of
        another record."""
        self.created = other.created
        self.ttl = other.ttl

    @abstractmethod
    def write(self, out):
        """Write data out"""
        raise NotImplementedError

    def to_string(self, other):
        """String representation with additional information"""
        arg = "%s/%s,%s" % (
            self.ttl, self.get_remaining_ttl(current_time_millis()), other)
        return DNSEntry.to_string(self, "record", arg)


class DNSAddress(DNSRecord):
    """A DNS address record"""

    def __init__(self, name, type_, class_, ttl, address):
        DNSRecord.__init__(self, name, type_, class_, ttl)
        self.address = address

    def write(self, out):
        """Used in constructing an outgoing packet"""
        out.write_string(self.address)

    def __eq__(self, other):
        """Tests equality on address"""
        return isinstance(other, DNSAddress) and self.address == other.address

    def __repr__(self):
        """String representation"""
        try:
            return str(socket.inet_ntoa(self.address))
        except Exception:  # TODO stop catching all Exceptions
            return str(self.address)


class DNSHinfo(DNSRecord):
    """A DNS host information record"""

    def __init__(self, name, type_, class_, ttl, cpu, os):
        DNSRecord.__init__(self, name, type_, class_, ttl)
        try:
            self.cpu = cpu.decode('utf-8')
        except AttributeError:
            self.cpu = cpu
        try:
            self.os = os.decode('utf-8')
        except AttributeError:
            self.os = os

    def write(self, out):
        """Used in constructing an outgoing packet"""
        out.write_character_string(self.cpu.encode('utf-8'))
        out.write_character_string(self.os.encode('utf-8'))

    def __eq__(self, other):
        """Tests equality on cpu and os"""
        return (isinstance(other, DNSHinfo) and
                self.cpu == other.cpu and self.os == other.os)

    def __repr__(self):
        """String representation"""
        return self.cpu + " " + self.os


class DNSPointer(DNSRecord):
    """A DNS pointer record"""

    def __init__(self, name, type_, class_, ttl, alias):
        DNSRecord.__init__(self, name, type_, class_, ttl)
        self.alias = alias

    def write(self, out):
        """Used in constructing an outgoing packet"""
        out.write_name(self.alias)

    def __eq__(self, other):
        """Tests equality on alias"""
        return isinstance(other, DNSPointer) and self.alias == other.alias

    def __repr__(self):
        """String representation"""
        return self.to_string(self.alias)


class DNSText(DNSRecord):
    """A DNS text record"""

    def __init__(self, name, type_, class_, ttl, text):
        assert isinstance(text, (bytes, type(None)))
        DNSRecord.__init__(self, name, type_, class_, ttl)
        self.text = text

    def write(self, out):
        """Used in constructing an outgoing packet"""
        out.write_string(self.text)

    def __eq__(self, other):
        """Tests equality on text"""
        return isinstance(other, DNSText) and self.text == other.text

    def __repr__(self):
        """String representation"""
        if len(self.text) > 10:
            return self.to_string(self.text[:7]) + "..."
        else:
            return self.to_string(self.text)


class DNSService(DNSRecord):
    """A DNS service record"""

    def __init__(self, name, type_, class_, ttl, priority, weight, port, server):
        DNSRecord.__init__(self, name, type_, class_, ttl)
        self.priority = priority
        self.weight = weight
        self.port = port
        self.server = server

    def write(self, out):
        """Used in constructing an outgoing packet"""
        out.write_short(self.priority)
        out.write_short(self.weight)
        out.write_short(self.port)
        out.write_name(self.server)

    def __eq__(self, other):
        """Tests equality on priority, weight, port and server"""
        return (isinstance(other, DNSService) and
                self.priority == other.priority and
                self.weight == other.weight and
                self.port == other.port and
                self.server == other.server)

    def __repr__(self):
        """String representation"""
        return self.to_string("%s:%s" % (self.server, self.port))


class DNSIncoming(QuietLogger):
    """Object representation of an incoming DNS packet"""

    def __init__(self, data):
        """Constructor from string holding bytes of packet"""
        self.offset = 0
        self.data = data
        self.questions = []
        self.answers = []
        self.id = 0
        self.flags = 0
        self.num_questions = 0
        self.num_answers = 0
        self.num_authorities = 0
        self.num_additionals = 0
        self.valid = False

        try:
            self.read_header()
            self.read_questions()
            self.read_others()
            self.valid = True

        except (IndexError, struct.error, IncomingDecodeError):
            self.log_exception_warning((
                'Choked at offset %d while unpacking %r', self.offset, data))

    def unpack(self, format_):
        length = struct.calcsize(format_)
        info = struct.unpack(
            format_, self.data[self.offset:self.offset + length])
        self.offset += length
        return info

    def read_header(self):
        """Reads header portion of packet"""
        (self.id, self.flags, self.num_questions, self.num_answers,
         self.num_authorities, self.num_additionals) = self.unpack(b'!6H')

    def read_questions(self):
        """Reads questions section of packet"""
        for i in range(self.num_questions):
            name = self.read_name()
            type_, class_ = self.unpack(b'!HH')

            question = DNSQuestion(name, type_, class_)
            self.questions.append(question)

    # def read_int(self):
    #     """Reads an integer from the packet"""
    #     return self.unpack(b'!I')[0]

    def read_character_string(self):
        """Reads a character string from the packet"""
        length = self.data[self.offset]
        self.offset += 1
        return self.read_string(length)

    def read_string(self, length):
        """Reads a string of a given length from the packet"""
        info = self.data[self.offset:self.offset + length]
        self.offset += length
        return info

    def read_unsigned_short(self):
        """Reads an unsigned short from the packet"""
        return self.unpack(b'!H')[0]

    def read_others(self):
        """Reads the answers, authorities and additionals section of the
        packet"""
        n = self.num_answers + self.num_authorities + self.num_additionals
        for i in range(n):
            domain = self.read_name()
            type_, class_, ttl, length = self.unpack(b'!HHiH')

            rec = None
            if type_ == _TYPE_A:
                rec = DNSAddress(
                    domain, type_, class_, ttl, self.read_string(4))
            elif type_ == _TYPE_CNAME or type_ == _TYPE_PTR:
                rec = DNSPointer(
                    domain, type_, class_, ttl, self.read_name())
            elif type_ == _TYPE_TXT:
                rec = DNSText(
                    domain, type_, class_, ttl, self.read_string(length))
            elif type_ == _TYPE_SRV:
                rec = DNSService(
                    domain, type_, class_, ttl,
                    self.read_unsigned_short(), self.read_unsigned_short(),
                    self.read_unsigned_short(), self.read_name())
            elif type_ == _TYPE_HINFO:
                rec = DNSHinfo(
                    domain, type_, class_, ttl,
                    self.read_character_string(), self.read_character_string())
            elif type_ == _TYPE_AAAA:
                rec = DNSAddress(
                    domain, type_, class_, ttl, self.read_string(16))
            else:
                # Try to ignore types we don't know about
                # Skip the payload for the resource record so the next
                # records can be parsed correctly
                self.offset += length

            if rec is not None:
                self.answers.append(rec)

    def is_query(self):
        """Returns true if this is a query"""
        return (self.flags & _FLAGS_QR_MASK) == _FLAGS_QR_QUERY

    def is_response(self):
        """Returns true if this is a response"""
        return (self.flags & _FLAGS_QR_MASK) == _FLAGS_QR_RESPONSE

    def read_utf(self, offset, length):
        """Reads a UTF-8 string of a given length from the packet"""
        return str(self.data[offset:offset + length], 'utf-8', 'replace')

    def read_name(self):
        """Reads a domain name from the packet"""
        result = ''
        off = self.offset
        next_ = -1
        first = off

        while True:
            length = self.data[off]
            off += 1
            if length == 0:
                break
            t = length & 0xC0
            if t == 0x00:
                result = ''.join((result, self.read_utf(off, length) + '.'))
                off += length
            elif t == 0xC0:
                if next_ < 0:
                    next_ = off + 1
                off = ((length & 0x3F) << 8) | self.data[off]
                if off >= first:
                    raise IncomingDecodeError(
                        "Bad domain name (circular) at %s" % (off,))
                first = off
            else:
                raise IncomingDecodeError("Bad domain name at %s" % (off,))

        if next_ >= 0:
            self.offset = next_
        else:
            self.offset = off

        return result


class DNSOutgoing(object):
    """Object representation of an outgoing packet"""

    def __init__(self, flags, multicast=True):
        self.finished = False
        self.id = 0
        self.multicast = multicast
        self.flags = flags
        self.names = {}
        self.data = []
        self.size = 12
        self.state = self.State.init

        self.questions = []
        self.answers = []
        self.authorities = []
        self.additionals = []

    def __repr__(self):
        return '<DNSOutgoing:{%s}>' % ', '.join([
            'multicast=%s' % self.multicast,
            'flags=%s' % self.flags,
            'questions=%s' % self.questions,
            'answers=%s' % self.answers,
            'authorities=%s' % self.authorities,
            'additionals=%s' % self.additionals,
        ])

    class State(enum.Enum):
        init = 0
        finished = 1

    def add_question(self, record):
        """Adds a question"""
        self.questions.append(record)

    def add_answer(self, inp, record):
        """Adds an answer"""
        if not record.suppressed_by(inp):
            self.add_answer_at_time(record, 0)

    def add_answer_at_time(self, record, now):
        """Adds an answer if it does not expire by a certain time"""
        if record is not None:
            if now == 0 or not record.is_expired(now):
                self.answers.append((record, now))

    def add_authoritative_answer(self, record):
        """Adds an authoritative answer"""
        self.authorities.append(record)

    def add_additional_answer(self, record):
        """ Adds an additional answer

        From: RFC 6763, DNS-Based Service Discovery, February 2013

        12.  DNS Additional Record Generation

           DNS has an efficiency feature whereby a DNS server may place
           additional records in the additional section of the DNS message.
           These additional records are records that the client did not
           explicitly request, but the server has reasonable grounds to expect
           that the client might request them shortly, so including them can
           save the client from having to issue additional queries.

           This section recommends which additional records SHOULD be generated
           to improve network efficiency, for both Unicast and Multicast DNS-SD
           responses.

        12.1.  PTR Records

           When including a DNS-SD Service Instance Enumeration or Selective
           Instance Enumeration (subtype) PTR record in a response packet, the
           server/responder SHOULD include the following additional records:

           o  The SRV record(s) named in the PTR rdata.
           o  The TXT record(s) named in the PTR rdata.
           o  All address records (type "A" and "AAAA") named in the SRV rdata.

        12.2.  SRV Records

           When including an SRV record in a response packet, the
           server/responder SHOULD include the following additional records:

           o  All address records (type "A" and "AAAA") named in the SRV rdata.

        """
        self.additionals.append(record)

    def pack(self, format_, value):
        self.data.append(struct.pack(format_, value))
        self.size += struct.calcsize(format_)

    def write_byte(self, value):
        """Writes a single byte to the packet"""
        self.pack(b'!c', bytes((value,)))

    def insert_short(self, index, value):
        """Inserts an unsigned short in a certain position in the packet"""
        self.data.insert(index, struct.pack(b'!H', value))
        self.size += 2

    def write_short(self, value):
        """Writes an unsigned short to the packet"""
        self.pack(b'!H', value)

    def write_int(self, value):
        """Writes an unsigned integer to the packet"""
        self.pack(b'!I', int(value))

    def write_string(self, value):
        """Writes a string to the packet"""
        assert isinstance(value, bytes)
        self.data.append(value)
        self.size += len(value)

    def write_utf(self, s):
        """Writes a UTF-8 string of a given length to the packet"""
        utfstr = s.encode('utf-8')
        length = len(utfstr)
        if length > 64:
            raise NamePartTooLongException
        self.write_byte(length)
        self.write_string(utfstr)

    def write_character_string(self, value):
        assert isinstance(value, bytes)
        length = len(value)
        if length > 256:
            raise NamePartTooLongException
        self.write_byte(length)
        self.write_string(value)

    def write_name(self, name):
        """
        Write names to packet

        18.14. Name Compression

        When generating Multicast DNS messages, implementations SHOULD use
        name compression wherever possible to compress the names of resource
        records, by replacing some or all of the resource record name with a
        compact two-byte reference to an appearance of that data somewhere
        earlier in the message [RFC1035].
        """

        # split name into each label
        parts = name.split('.')
        if not parts[-1]:
            parts.pop()

        # construct each suffix
        name_suffices = ['.'.join(parts[i:]) for i in range(len(parts))]

        # look for an existing name or suffix
        for count, sub_name in enumerate(name_suffices):
            if sub_name in self.names:
                break
        else:
            count += 1

        # note the new names we are saving into the packet
        for suffix in name_suffices[:count]:
            self.names[suffix] = self.size + len(name) - len(suffix) - 1

        # write the new names out.
        for part in parts[:count]:
            self.write_utf(part)

        # if we wrote part of the name, create a pointer to the rest
        if count != len(name_suffices):
            # Found substring in packet, create pointer
            index = self.names[name_suffices[count]]
            self.write_byte((index >> 8) | 0xC0)
            self.write_byte(index & 0xFF)
        else:
            # this is the end of a name
            self.write_byte(0)

    def write_question(self, question):
        """Writes a question to the packet"""
        self.write_name(question.name)
        self.write_short(question.type)
        self.write_short(question.class_)

    def write_record(self, record, now):
        """Writes a record (answer, authoritative answer, additional) to
        the packet"""
        if self.state == self.State.finished:
            return 1

        start_data_length, start_size = len(self.data), self.size
        self.write_name(record.name)
        self.write_short(record.type)
        if record.unique and self.multicast:
            self.write_short(record.class_ | _CLASS_UNIQUE)
        else:
            self.write_short(record.class_)
        if now == 0:
            self.write_int(record.ttl)
        else:
            self.write_int(record.get_remaining_ttl(now))
        index = len(self.data)

        # Adjust size for the short we will write before this record
        self.size += 2
        record.write(self)
        self.size -= 2

        length = sum((len(d) for d in self.data[index:]))
        # Here is the short we adjusted for
        self.insert_short(index, length)

        # if we go over, then rollback and quit
        if self.size > _MAX_MSG_ABSOLUTE:
            while len(self.data) > start_data_length:
                self.data.pop()
            self.size = start_size
            self.state = self.State.finished
            return 1
        return 0

    def packet(self):
        """Returns a string containing the packet's bytes

        No further parts should be added to the packet once this
        is done."""

        overrun_answers, overrun_authorities, overrun_additionals = 0, 0, 0

        if self.state != self.State.finished:
            for question in self.questions:
                self.write_question(question)
            for answer, time_ in self.answers:
                overrun_answers += self.write_record(answer, time_)
            for authority in self.authorities:
                overrun_authorities += self.write_record(authority, 0)
            for additional in self.additionals:
                overrun_additionals += self.write_record(additional, 0)
            self.state = self.State.finished

            self.insert_short(0, len(self.additionals) - overrun_additionals)
            self.insert_short(0, len(self.authorities) - overrun_authorities)
            self.insert_short(0, len(self.answers) - overrun_answers)
            self.insert_short(0, len(self.questions))
            self.insert_short(0, self.flags)
            if self.multicast:
                self.insert_short(0, 0)
            else:
                self.insert_short(0, self.id)
        return b''.join(self.data)


class DNSCache(object):
    """A cache of DNS entries"""

    def __init__(self):
        self.cache = {}

    def add(self, entry):
        """Adds an entry"""
        self.cache.setdefault(entry.key, []).append(entry)

    def remove(self, entry):
        """Removes an entry"""
        try:
            list_ = self.cache[entry.key]
            list_.remove(entry)
        except (KeyError, ValueError):
            pass

    def get(self, entry):
        """Gets an entry by key.  Will return None if there is no
        matching entry."""
        try:
            list_ = self.cache[entry.key]
            for cached_entry in list_:
                if entry.__eq__(cached_entry):
                    return cached_entry
        except (KeyError, ValueError):
            return None

    def get_by_details(self, name, type_, class_):
        """Gets an entry by details.  Will return None if there is
        no matching entry."""
        entry = DNSEntry(name, type_, class_)
        return self.get(entry)

    def entries_with_name(self, name):
        """Returns a list of entries whose key matches the name."""
        try:
            return self.cache[name.lower()]
        except KeyError:
            return []

    def current_entry_with_name_and_alias(self, name, alias):
        now = current_time_millis()
        for record in self.entries_with_name(name):
            if (record.type == _TYPE_PTR and
                    not record.is_expired(now) and
                    record.alias == alias):
                return record

    def entries(self):
        """Returns a list of all entries"""
        if not self.cache:
            return []
        else:
            # avoid size change during iteration by copying the cache
            values = list(self.cache.values())
            return reduce(lambda a, b: a + b, values)


class MCListener(asyncio.Protocol, QuietLogger):
    """A MCListener is used by this module to listen on the multicast
    group to which DNS messages are sent, allowing the implementation
    to cache information as it arrives."""

    def __init__(self, zc, af, senders):
        """
        :param zc: Zeroconf instance
        :param af: address familly
        :param senders: list of sending socket transports
        """
        asyncio.Protocol.__init__(self)
        self.zc = zc
        self.af = af
        self.senders = senders
        self.transport = None

    def connection_made(self, transport):
        self.transport = transport

    def datagram_received(self, data, addrs):
        try:
            assert len(addrs) in [2, 4], "What network protocol is that?"
            if len(addrs) == 2:
                addr, port = addrs
            else:
                addr, port, flow, scope = addrs
        except AssertionError:
            self.log_exception_warning()
            return

        log.debug('Received from %r:%r: %r ', addr, port, data)

        msg = DNSIncoming(data)
        if not msg.valid:
            return

        if msg.is_query():
            resp_addr = _MDNS_ADDR if self.af == socket.AF_INET else _MDNS6_ADDR

            # Always multicast responses
            if port == _MDNS_PORT:
                self.zc.handle_query(msg, resp_addr, _MDNS_PORT)

            # If it's not a multicast query, reply via unicast
            # and multicast
            elif port == _DNS_PORT:
                self.zc.handle_query(msg, addr, port)
                self.zc.handle_query(msg, resp_addr, _MDNS_PORT)

        else:
            self.zc.handle_response(msg)

    def close(self):
        if self.transport:
            self.transport.close()

        for sender in self.senders:
            sender.close()

    def sendto(self, data, destination):
        for sender in self.senders:
            sender.sendto(data, destination)


class Reaper(object):
    """A Reaper is used by this module to remove cache entries that
    have expired."""

    def __init__(self, zc):
        self.zc = zc
        self.task = asyncio.ensure_future(self.run())

    async def run(self):
        while True:
            await asyncio.sleep(10)
            now = current_time_millis()
            for record in self.zc.cache.entries():
                if record.is_expired(now):
                    self.zc.update_record(now, record)
                    self.zc.cache.remove(record)


class ServiceBrowser(object):
    """Used to browse for a service of a specific type.

    The listener object will have its add_service() and
    remove_service() methods called when this browser
    discovers changes in the services availability."""

    def __init__(self, zc, type_, handlers=None, listener=None):
        """Creates a browser for a specific type"""
        assert handlers or listener, 'You need to specify at least one handler'
        if not type_.endswith(service_type_name(type_)):
            raise BadTypeInNameException

        self.zc = zc
        self.type = type_
        self.services = {}
        self.next_time = 100
        self.delay = _BROWSER_TIME
        self.done = False

        if hasattr(handlers, 'add_service'):
            listener = handlers
            handlers = None

        if handlers and not isinstance(handlers, list):
            self.handlers = [handlers]
        else:
            self.handlers = handlers or []
        self.listener = listener or None
        self.task = asyncio.ensure_future(self.run())

    def update_record(self, zc, now, record):
        """Callback invoked by Zeroconf when new information arrives.

        Updates information required by browser in the Zeroconf cache."""

        if record.type == _TYPE_PTR and record.name == self.type:
            expired = record.is_expired(now)
            service_key = record.alias.lower()
            try:
                old_record = self.services[service_key]
            except KeyError:
                if not expired:
                    self.services[service_key] = record
                    if self.listener:
                        self.listener.add_service(self.zc, self.type, record.alias)
                    for hdlr in self.handlers:
                        hdlr(self.zc, self.type, record.alias, ServiceStateChange.Added)

            else:
                if not expired:
                    old_record.reset_ttl(record)
                else:
                    del self.services[service_key]
                    if self.listener:
                        self.listener.remove_service(self.zc, self.type, record.alias)
                    for hdlr in self.handlers:
                        hdlr(self.zc, self.type, record.alias, ServiceStateChange.Removed)
                    return

            expires = record.get_expiration_time(75)
            if expires < self.next_time:
                self.next_time = expires

    def cancel(self):
        self.done = True
        self.task.cancel()
        self.zc.remove_listener(self)

    async def run(self):
        self.zc.add_listener(self, DNSQuestion(self.type, _TYPE_PTR, _CLASS_IN))

        while True:
            now = current_time_millis()
            if self.next_time > now:
                await asyncio.sleep(round((self.next_time - now) / 1000.0, 2))
            if self.zc.done or self.done:
                return
            now = current_time_millis()
            if self.next_time <= now:
                out = DNSOutgoing(_FLAGS_QR_QUERY)
                out.add_question(DNSQuestion(self.type, _TYPE_PTR, _CLASS_IN))
                for record in self.services.values():
                    if not record.is_expired(now):
                        out.add_answer_at_time(record, now)
                self.zc.send(out)
                self.next_time = now + self.delay
                self.delay = min(20 * 1000, self.delay * 2)


class ServiceInfo(object):
    """Service information"""

    def __init__(self, type_, name, *, address=None, address6=None, port=None, weight=0,
                 priority=0, properties=None, server=None):
        """Create a service description.

        type_: fully qualified service type name
        name: fully qualified service name
        address: IP address as unsigned short, network byte order
        address6: IPv6 address as 16 byte, network byte order
        port: port that the service runs on
        weight: weight of the service
        priority: priority of the service
        properties: dictionary of properties (or a string holding the
                    bytes for the text field)
        server: fully qualified name for service host (defaults to name)"""

        if not type_.endswith(service_type_name(name)):
            raise BadTypeInNameException
        self.type = type_
        self.name = name
        self.address = address
        self.address6 = address6
        self.port = port
        self.weight = weight
        self.priority = priority
        if server:
            self.server = server
        else:
            self.server = name
        self._properties = {}
        self._set_properties(properties)

    @property
    def properties(self):
        return self._properties

    def _set_properties(self, properties):
        """Sets properties and text of this info from a dictionary"""
        if isinstance(properties, dict):
            self._properties = properties
            list_ = []
            result = b''
            for key, value in properties.items():
                if isinstance(key, str):
                    key = key.encode('utf-8')

                if value is None:
                    suffix = b''
                elif isinstance(value, str):
                    suffix = value.encode('utf-8')
                elif isinstance(value, bytes):
                    suffix = value
                elif isinstance(value, int):
                    if value:
                        suffix = b'true'
                    else:
                        suffix = b'false'
                else:
                    suffix = b''
                list_.append(b'='.join((key, suffix)))
            for item in list_:
                result = b''.join((result, bytes((len(item),)), item))
            self.text = result
        else:
            self.text = properties

    def _set_text(self, text):
        """Sets properties and text given a text field"""
        self.text = text
        result = {}
        end = len(text)
        index = 0
        strs = []
        while index < end:
            length = text[index]
            index += 1
            strs.append(text[index:index + length])
            index += length

        for s in strs:
            parts = s.split(b'=', 1)
            try:
                key, value = parts
            except ValueError:
                # No equals sign at all
                key = s
                value = False
            else:
                if value == b'true':
                    value = True
                elif value == b'false' or not value:
                    value = False

            # Only update non-existent properties
            if key and result.get(key) is None:
                result[key] = value

        self._properties = result

    def get_name(self):
        """Name accessor"""
        if self.type is not None and self.name.endswith("." + self.type):
            return self.name[:len(self.name) - len(self.type) - 1]
        return self.name

    def update_record(self, zc, now, record):
        """Updates service information from a DNS record"""
        if record is not None and not record.is_expired(now):
            if record.type == _TYPE_A:
                # if record.name == self.name:
                if record.name == self.server:
                    self.address = record.address
            elif record.type == _TYPE_AAAA:
                # if record.name == self.name:
                if record.name == self.server:
                    self.address6 = record.address
            elif record.type == _TYPE_SRV:
                if record.name == self.name:
                    self.server = record.server
                    self.port = record.port
                    self.weight = record.weight
                    self.priority = record.priority
                    # self.address = None
                    self.update_record(
                        zc, now, zc.cache.get_by_details(
                            self.server, _TYPE_A, _CLASS_IN))
                    self.update_record(
                        zc, now, zc.cache.get_by_details(
                            self.server, _TYPE_AAAA, _CLASS_IN))
            elif record.type == _TYPE_TXT:
                if record.name == self.name:
                    self._set_text(record.text)

    def all_set(self, zc):
        """
        Return True if all wanted fields have been retrieved.
        Only care for an address type if corresponding AF is configured.
        """
        return (
            self.server is not None and self.text is not None and
            (self.address is not None if socket.AF_INET in zc.protocols else True) and
            (self.address6 is not None if socket.AF_INET6 in zc.protocols else True)
        )

    async def request(self, zc, timeout):
        """Coroutine: Returns true if the service could be discovered on the
        network, and updates this object with details discovered.
        """
        now = current_time_millis()
        delay = _LISTENER_TIME
        next_ = now + delay
        last = now + timeout

        record_types_for_check_cache = [
            (_TYPE_SRV, _CLASS_IN),
            (_TYPE_TXT, _CLASS_IN),
        ]
        if self.server is not None:
            record_types_for_check_cache.append((_TYPE_A, _CLASS_IN))
            record_types_for_check_cache.append((_TYPE_AAAA, _CLASS_IN))
        for record_type in record_types_for_check_cache:
            cached = zc.cache.get_by_details(self.name, *record_type)
            if cached:
                self.update_record(zc, now, cached)

        if self.all_set(zc):
            # All possible values have been retrieved
            return True

        if timeout == 0:
            # Timeout is set to zero: only return what was available in cache
            return False

        try:
            zc.add_listener(self, DNSQuestion(self.name, _TYPE_ANY, _CLASS_IN))
            while not self.all_set(zc):
                if last <= now:
                    return False
                if next_ <= now:
                    out = DNSOutgoing(_FLAGS_QR_QUERY)
                    out.add_question(
                        DNSQuestion(self.name, _TYPE_SRV, _CLASS_IN))
                    out.add_answer_at_time(
                        zc.cache.get_by_details(
                            self.name, _TYPE_SRV, _CLASS_IN), now)

                    out.add_question(
                        DNSQuestion(self.name, _TYPE_TXT, _CLASS_IN))
                    out.add_answer_at_time(
                        zc.cache.get_by_details(
                            self.name, _TYPE_TXT, _CLASS_IN), now)

                    if self.server is not None:
                        # Always ask for both AF; does not cost much!
                        out.add_question(
                            DNSQuestion(self.server, _TYPE_A, _CLASS_IN))
                        out.add_answer_at_time(
                            zc.cache.get_by_details(
                                self.server, _TYPE_A, _CLASS_IN), now)
                        out.add_question(
                            DNSQuestion(self.server, _TYPE_AAAA, _CLASS_IN))
                        out.add_answer_at_time(
                            zc.cache.get_by_details(
                                self.server, _TYPE_AAAA, _CLASS_IN), now)
                    zc.send(out)
                    next_ = now + delay
                    delay *= 2
                await asyncio.sleep((min(next_, last) - now) / 1000.0)
                now = current_time_millis()
        except Exception as e:
            log.debug("Request failed: {}".format(e))
        finally:
            zc.remove_listener(self)
        return True

    def __eq__(self, other):
        """Tests equality of service name"""
        return isinstance(other, ServiceInfo) and other.name == self.name

    def __ne__(self, other):
        """Non-equality test"""
        return not self.__eq__(other)

    def __repr__(self):
        """String representation"""
        return '%s(%s)' % (
            type(self).__name__,
            ', '.join(
                '%s=%r' % (name, getattr(self, name))
                for name in (
                    'type', 'name', 'address', 'port', 'weight', 'priority',
                    'server', 'properties',
                )
            )
        )


class ZeroconfServiceTypes(object):
    """
    Return all of the advertised services on any local networks
    """

    def __init__(self):
        self.found_services = set()

    def add_service(self, zc, type_, name):
        self.found_services.add(name)

    def remove_service(self, zc, type_, name):
        pass

    @classmethod
    async def find(cls, zc, timeout=5):
        """
        Return all of the advertised services on any local networks.

        :param zc: Zeroconf() instance.  Pass in an instance running
        :param timeout: seconds to wait for any responses
        :return: tuple of service type strings
        """
        listener = cls()
        browser = ServiceBrowser(
            zc, '_services._dns-sd._udp.local.', listener=listener)

        # wait for responses
        await asyncio.sleep(timeout)

        browser.cancel()

        return tuple(sorted(listener.found_services))


def normalize_interface_choice(iface: Union[str, None]) -> List[str]:
    """
    Normalize interface provided into a list of addresses
    :param iface: name of a valid interface or None
    """
    interfaces = netifaces.interfaces()

    if iface:
        if iface not in interfaces:
            raise ValueError('invalid interface name: {}'.format(iface))
        interfaces = [iface]

    return interfaces


def new_inet_socket(port: int = _MDNS_PORT) -> socket.socket:
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

    # SO_REUSEADDR should be equivalent to SO_REUSEPORT for
    # multicast UDP sockets (p 731, "TCP/IP Illustrated,
    # Volume 2"), but some BSD-derived systems require
    # SO_REUSEPORT to be specified explicity.  Also, not all
    # versions of Python have SO_REUSEPORT available.
    # Catch OSError and socket.error for kernel versions <3.9 because lacking
    # SO_REUSEPORT support.
    try:
        reuseport = socket.SO_REUSEPORT
    except AttributeError:
        pass
    else:
        try:
            s.setsockopt(socket.SOL_SOCKET, reuseport, 1)
        except (OSError, socket.error) as err:
            # OSError on python 3, socket.error on python 2
            if not err.errno == errno.ENOPROTOOPT:
                raise

    if port is _MDNS_PORT:
        # OpenBSD needs the ttl and loop values for the IP_MULTICAST_TTL and
        # IP_MULTICAST_LOOP socket options as an unsigned char.
        ttl = struct.pack(b'B', 255)
        s.setsockopt(socket.IPPROTO_IP, socket.IP_MULTICAST_TTL, ttl)
        # loop = struct.pack(b'B', 1)
        # s.setsockopt(socket.IPPROTO_IP, socket.IP_MULTICAST_LOOP, loop)

    s.bind(('', port))
    return s


def setup_inet(interfaces: List[str]):
    """
    Return new sockets for sending and receiving
    :param interfaces:
    :return:
    """
    # Create listening socket
    listener = new_inet_socket()

    addresses = []
    for interface in interfaces:
        addr_list = netifaces.ifaddresses(interface)
        if netifaces.AF_INET in addr_list:
            addresses.append((interface, addr_list[netifaces.AF_INET][0]['addr']))

    if not addresses:
        raise ValueError('No interface for IPv4')

    addrinfo = socket.getaddrinfo(_MDNS_ADDR, None)[0]
    group_bin = socket.inet_pton(addrinfo[0], addrinfo[4][0])

    # List of sender sockets
    senders = []
    for interface, addr in addresses:
        bind_addr = socket.inet_aton(addr)
        try:
            log.info('binding on %s -> %s', interface, addr)
            listener.setsockopt(socket.IPPROTO_IP, socket.IP_ADD_MEMBERSHIP, group_bin + bind_addr)
        except socket.error as e:
            _errno = get_errno(e)
            if _errno == errno.EADDRINUSE:
                log.info('Address in use when adding %s to multicast group', interface)
            elif _errno == errno.EADDRNOTAVAIL:
                log.info('Address not available when adding %s to multicast', interface)
                continue
            elif _errno == errno.EINVAL:
                log.info('Interface %s does not support multicast', interface)
                continue
            else:
                raise

        sender = new_inet_socket()
        sender.setsockopt(socket.IPPROTO_IP, socket.IP_MULTICAST_IF, bind_addr)
        senders.append(sender)

    return listener, senders


def new_inet6_socket(port: int = _MDNS_PORT) -> socket.socket:
    s = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)
    s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

    # SO_REUSEADDR should be equivalent to SO_REUSEPORT for
    # multicast UDP sockets (p 731, "TCP/IP Illustrated,
    # Volume 2"), but some BSD-derived systems require
    # SO_REUSEPORT to be specified explicity.  Also, not all
    # versions of Python have SO_REUSEPORT available.
    # Catch OSError and socket.error for kernel versions <3.9 because lacking
    # SO_REUSEPORT support.
    try:
        reuseport = socket.SO_REUSEPORT
    except AttributeError:
        pass
    else:
        try:
            s.setsockopt(socket.SOL_SOCKET, reuseport, 1)
        except (OSError, socket.error) as err:
            # OSError on python 3, socket.error on python 2
            if not err.errno == errno.ENOPROTOOPT:
                raise

    if port is _MDNS_PORT:
        ttl = struct.pack('@i', 2)
        s.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_MULTICAST_HOPS, ttl)

    s.bind(('', port))
    return s


def setup_inet6(interfaces: List[str]):
    """
    Return new sockets for sending and receiving
    :param interfaces:
    :return:
    """
    # Create listening socket
    listener = new_inet6_socket()

    indexes = []
    for interface in interfaces:
        addr_list = netifaces.ifaddresses(interface)
        if netifaces.AF_INET6 in addr_list:
            addr = addr_list[netifaces.AF_INET6][0]["addr"]
            idx = socket.getaddrinfo(addr, _MDNS_PORT)[0][4][3]
            indexes.append((interface, idx))

    if not indexes:
        raise ValueError('No interface for IPv6')

    addrinfo = socket.getaddrinfo(_MDNS6_ADDR, None)[0]
    group_bin = socket.inet_pton(addrinfo[0], addrinfo[4][0])

    # List of sender sockets
    senders = []
    for interface, idx in indexes:
        bind_idx = struct.pack('@I', idx)
        try:
            log.info('binding on %s -> %s', interface, idx)
            listener.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_JOIN_GROUP, group_bin + bind_idx)
        except socket.error as e:
            _errno = get_errno(e)
            if _errno == errno.EADDRINUSE:
                log.info('Address in use when adding %s to multicast group', interface)
            elif _errno == errno.EADDRNOTAVAIL:
                log.info('Address not available when adding %s to multicast', interface)
                continue
            elif _errno == errno.EINVAL:
                log.info('Interface %s does not support multicast', interface)
                continue
            else:
                raise

        sender = new_inet6_socket()
        sender.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_MULTICAST_IF, bind_idx)
        senders.append(sender)

    return listener, senders


def get_errno(e):
    assert isinstance(e, socket.error)
    return e.args[0]


class Zeroconf(QuietLogger):
    """Implementation of Zeroconf Multicast DNS Service Discovery

    Supports registration, unregistration, queries and browsing.
    """

    def __init__(
            self,
            loop,
            address_family=[netifaces.AF_INET, netifaces.AF_INET6],
            iface=None
    ):
        """Creates an instance of the Zeroconf class, establishing
        multicast communications, listening and reaping threads.

        :type iface: :class string Name of the interface
        """
        self.loop = loop
        self.protocols = {}

        self.interfaces = normalize_interface_choice(iface)

        self._init = self.loop.create_task(self.initialize(address_family))
        self.listeners = []
        self.browsers = {}
        self.services = {}
        self.servicetypes = {}

        self.cache = DNSCache()
        self.reaper = Reaper(self)

        self.debug = None
        self._GLOBAL_DONE = False

    @property
    def done(self):
        return self._GLOBAL_DONE

    async def initialize(self, address_family):
        """
        Initialize communication for all provided interfaces
        :param address_family: list of AF
        """
        for af in address_family:
            try:
                if af == netifaces.AF_INET:
                    listener, senders = setup_inet(self.interfaces)
                elif af == netifaces.AF_INET6:
                    listener, senders = setup_inet6(self.interfaces)
                else:
                    raise ValueError('Only IPv4 and IPv6 are supported')
            except Exception as e:
                log.error('initializing comm af %s on interfaces %s: %s', af, self.interfaces, e)
                continue

            if listener:
                try:
                    # Create an asyncio datagram transport for each sender (avoid blocking)
                    sender_protocols = []
                    for sender in senders:
                        transport, _ = await self.loop.create_datagram_endpoint(
                            asyncio.DatagramProtocol,
                            sock=sender,
                        )
                        sender_protocols.append(transport)

                    # Create an asyncio datagram protocol for receiving data
                    _, protocol = await self.loop.create_datagram_endpoint(
                        partial(MCListener, self, af, sender_protocols),
                        sock=listener,
                    )
                    self.protocols[af] = protocol
                except asyncio.CancelledError:
                    # Task was cancelled while we were creating endpoint...
                    try:
                        listener.close()
                    except socket.error:
                        pass
                    for sender in senders:
                        try:
                            sender.close()
                        except socket.error:
                            pass
                    raise

    async def get_service_info(self, type_, name, timeout=3000):
        """Returns network's service information for a particular
        name and type, or None if no service matches by the timeout,
        which defaults to 3 seconds."""
        info = ServiceInfo(type_, name)
        await info.request(self, timeout)
        return info

    def add_service_listener(self, type_, listener):
        """Adds a listener for a particular service type.  This object
        will then have its update_record method called when information
        arrives for that type."""
        self.remove_service_listener(listener)
        self.browsers[listener] = ServiceBrowser(self, type_, listener)

    def remove_service_listener(self, listener):
        """Removes a listener from the set that is currently listening."""
        if listener in self.browsers:
            self.browsers[listener].cancel()
            del self.browsers[listener]

    def remove_all_service_listeners(self):
        """Removes a listener from the set that is currently listening."""
        for listener in [k for k in self.browsers]:
            self.remove_service_listener(listener)

    async def register_service(self, info, ttl=_DNS_TTL, allow_name_change=False):
        """Registers service information to the network with a default TTL
        of 60 seconds.  Zeroconf will then respond to requests for
        information for that service.  The name of the service may be
        changed if needed to make it unique on the network."""
        await self.check_service(info, allow_name_change)
        self.services[info.name.lower()] = info
        if info.type in self.servicetypes:
            self.servicetypes[info.type] += 1
        else:
            self.servicetypes[info.type] = 1
        now = current_time_millis()
        next_time = now
        i = 0
        while i < 3:
            if now < next_time:
                await asyncio.sleep((next_time - now) / 1000.0)
                now = current_time_millis()
                continue
            out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)
            out.add_answer_at_time(
                DNSPointer(info.type, _TYPE_PTR, _CLASS_IN, ttl, info.name), 0)
            out.add_answer_at_time(
                DNSService(info.name, _TYPE_SRV, _CLASS_IN,
                           ttl, info.priority, info.weight, info.port,
                           info.server), 0)

            out.add_answer_at_time(
                DNSText(info.name, _TYPE_TXT, _CLASS_IN, ttl, info.text), 0)
            if info.address:
                out.add_answer_at_time(
                    DNSAddress(info.server, _TYPE_A, _CLASS_IN,
                               ttl, info.address), 0)
            if info.address6:
                out.add_answer_at_time(
                    DNSAddress(info.server, _TYPE_AAAA, _CLASS_IN,
                               ttl, info.address6), 0)
            self.send(out)
            i += 1
            next_time += _REGISTER_TIME

    async def unregister_service(self, info):
        """Unregister a service."""
        try:
            del self.services[info.name.lower()]
            if self.servicetypes[info.type] > 1:
                self.servicetypes[info.type] -= 1
            else:
                del self.servicetypes[info.type]
        except Exception as e:  # TODO stop catching all Exceptions
            log.exception('Unknown error, possibly benign: %r', e)
        now = current_time_millis()
        next_time = now
        i = 0
        while i < 3:
            if now < next_time:
                await asyncio.sleep((next_time - now) / 1000.0)
                now = current_time_millis()
                continue
            out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)
            out.add_answer_at_time(
                DNSPointer(info.type, _TYPE_PTR, _CLASS_IN, 0, info.name), 0)
            out.add_answer_at_time(
                DNSService(info.name, _TYPE_SRV, _CLASS_IN, 0,
                           info.priority, info.weight, info.port, info.name), 0)
            out.add_answer_at_time(
                DNSText(info.name, _TYPE_TXT, _CLASS_IN, 0, info.text), 0)

            if info.address:
                out.add_answer_at_time(
                    DNSAddress(info.server, _TYPE_A, _CLASS_IN, 0,
                               info.address), 0)

            if info.address6:
                out.add_answer_at_time(
                    DNSAddress(info.server, _TYPE_AAAA, _CLASS_IN, 0,
                               info.address6), 0)
            self.send(out)
            i += 1
            next_time += _UNREGISTER_TIME

    async def unregister_all_services(self):
        """Unregister all registered services."""
        if len(self.services) > 0:
            now = current_time_millis()
            next_time = now
            i = 0
            while i < 3:
                if now < next_time:
                    await asyncio.sleep((next_time - now) / 1000.0)
                    now = current_time_millis()
                    continue
                out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)
                for info in self.services.values():
                    out.add_answer_at_time(DNSPointer(
                        info.type, _TYPE_PTR, _CLASS_IN, 0, info.name), 0)
                    out.add_answer_at_time(DNSService(
                        info.name, _TYPE_SRV, _CLASS_IN, 0,
                        info.priority, info.weight, info.port, info.server), 0)
                    out.add_answer_at_time(DNSText(
                        info.name, _TYPE_TXT, _CLASS_IN, 0, info.text), 0)
                    if info.address:
                        out.add_answer_at_time(DNSAddress(
                            info.server, _TYPE_A, _CLASS_IN, 0,
                            info.address), 0)
                    if info.address6:
                        out.add_answer_at_time(DNSAddress(
                            info.server, _TYPE_AAAA, _CLASS_IN, 0,
                            info.address6), 0)
                self.send(out)
                i += 1
                next_time += _UNREGISTER_TIME

    async def check_service(self, info, allow_name_change):
        """Checks the network for a unique service name, modifying the
        ServiceInfo passed in if it is not unique."""

        # This is kind of funky because of the subtype based tests
        # need to make subtypes a first class citizen
        service_name = service_type_name(info.name)
        if not info.type.endswith(service_name):
            raise BadTypeInNameException

        instance_name = info.name[:-len(service_name) - 1]
        next_instance_number = 2

        now = current_time_millis()
        next_time = now
        i = 0
        while i < 3:
            # check for a name conflict
            while self.cache.current_entry_with_name_and_alias(
                    info.type, info.name):
                if not allow_name_change:
                    raise NonUniqueNameException

                # change the name and look for a conflict
                info.name = '%s-%s.%s' % (
                    instance_name, next_instance_number, info.type)
                next_instance_number += 1
                service_type_name(info.name)
                next_time = now
                i = 0

            if now < next_time:
                await asyncio.sleep((next_time - now) / 1000.0)
                now = current_time_millis()
                continue

            out = DNSOutgoing(_FLAGS_QR_QUERY | _FLAGS_AA)
            self.debug = out
            out.add_question(DNSQuestion(info.type, _TYPE_PTR, _CLASS_IN))
            out.add_authoritative_answer(DNSPointer(
                info.type, _TYPE_PTR, _CLASS_IN, _DNS_TTL, info.name))
            self.send(out)
            i += 1
            next_time += _CHECK_TIME

    def add_listener(self, listener, question):
        """Adds a listener for a given question.  The listener will have
        its update_record method called when information is available to
        answer the question."""
        now = current_time_millis()
        self.listeners.append(listener)
        if question is not None:
            for record in self.cache.entries_with_name(question.name):
                if question.answered_by(record) and not record.is_expired(now):
                    listener.update_record(self, now, record)

    def remove_listener(self, listener):
        """Removes a listener."""
        if listener in self.listeners:
            self.listeners.remove(listener)
        else:
            log.warning('cannot remove listener %s: not found', listener)

    def update_record(self, now, rec):
        """Used to notify listeners of new information that has updated
        a record."""
        for listener in self.listeners:
            listener.update_record(self, now, rec)

    def handle_response(self, msg):
        """Deal with incoming response packets.  All answers
        are held in the cache, and listeners are notified."""
        now = current_time_millis()
        for record in msg.answers:
            expired = record.is_expired(now)
            if record in self.cache.entries():
                if expired:
                    self.cache.remove(record)
                else:
                    entry = self.cache.get(record)
                    if entry is not None:
                        entry.reset_ttl(record)
            else:
                self.cache.add(record)

        for record in msg.answers:
            self.update_record(now, record)

    def handle_query(self, msg, addr, port):
        """Deal with incoming query packets.  Provides a response if
        possible."""
        out = None

        # Support unicast client responses
        #
        if port != _MDNS_PORT:
            out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA, multicast=False)
            for question in msg.questions:
                out.add_question(question)

        for question in msg.questions:
            if question.type == _TYPE_PTR:
                if question.name == "_services._dns-sd._udp.local.":
                    for stype in self.servicetypes.keys():
                        if out is None:
                            out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)
                        out.add_answer(msg, DNSPointer(
                            "_services._dns-sd._udp.local.", _TYPE_PTR,
                            _CLASS_IN, _DNS_TTL, stype))
                for service in self.services.values():
                    if question.name == service.type:
                        if out is None:
                            out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)
                        out.add_answer(msg, DNSPointer(
                            service.type, _TYPE_PTR,
                            _CLASS_IN, _DNS_TTL, service.name))
            else:
                try:
                    if out is None:
                        out = DNSOutgoing(_FLAGS_QR_RESPONSE | _FLAGS_AA)

                    # Answer A record queries for any service addresses we know
                    if question.type in (_TYPE_A, _TYPE_ANY):
                        for service in self.services.values():
                            if service.server == question.name.lower():
                                out.add_answer(msg, DNSAddress(
                                    question.name, _TYPE_A,
                                    _CLASS_IN | _CLASS_UNIQUE,
                                    _DNS_TTL, service.address))

                    service = self.services.get(question.name.lower(), None)
                    if not service:
                        continue

                    if question.type in (_TYPE_SRV, _TYPE_ANY):
                        out.add_answer(msg, DNSService(
                            question.name, _TYPE_SRV, _CLASS_IN | _CLASS_UNIQUE,
                            _DNS_TTL, service.priority, service.weight,
                            service.port, service.server))
                    if question.type in (_TYPE_TXT, _TYPE_ANY):
                        out.add_answer(msg, DNSText(
                            question.name, _TYPE_TXT, _CLASS_IN | _CLASS_UNIQUE,
                            _DNS_TTL, service.text))
                    if question.type == _TYPE_SRV:
                        out.add_additional_answer(DNSAddress(
                            service.server, _TYPE_A, _CLASS_IN | _CLASS_UNIQUE,
                            _DNS_TTL, service.address))
                except Exception:  # TODO stop catching all Exceptions
                    self.log_exception_warning()

        if out is not None and out.answers:
            out.id = msg.id
            self.send(out, addr, port)

    def send(self, out, addr=None, port=None):
        """Sends an outgoing packet."""
        packet = out.packet()
        if len(packet) > _MAX_MSG_ABSOLUTE:
            self.log_warning_once("Dropping %r over-sized packet (%d bytes) %r",
                                  out, len(packet), packet)
            return
        log.debug('Sending %r (%d bytes) as %r...', out, len(packet), packet)
        for af, proto in self.protocols.items():
            if addr:
                addrfam = socket.getaddrinfo(addr, None)[0][0]
                if addrfam != af:
                    continue

            if self._GLOBAL_DONE:
                return
            try:
                if af == netifaces.AF_INET:
                    proto.sendto(packet, (addr or _MDNS_ADDR, port or _MDNS_PORT))
                else:
                    proto.sendto(packet, (addr or _MDNS6_ADDR, port or _MDNS_PORT))
            except socket.error:
                # on send errors, log the exception and keep going
                self.log_exception_warning()

    async def close(self):
        """Ends the background threads, and prevent this instance from
        servicing further queries."""
        if not self._GLOBAL_DONE:
            self._GLOBAL_DONE = True
            # remove service listeners
            self.remove_all_service_listeners()
            await self.unregister_all_services()

            # shutdown recv socket and thread
            for proto in self.protocols.values():
                proto.close()

            if not self._init.done():
                self._init.cancel()
                try:
                    await self._init
                except asyncio.CancelledError:
                    pass

            # shutdown the rest
            self.reaper.task.cancel()
