"""
Build tool to create Docker images. This is another implementation of the builder different than the
one provided by the official repository. This tool is intended to give the user more flexibility for
the creation of an image as opposed to what is being provided by the build command provided in
docker
"""
import argparse
import logging
import sys
import time

from docker_build import \
    initialise_logging, \
    log, \
    build,\
    __version__, \
    DEFAULT_KEEP_CONTAINERS
from docker_build.exception import \
    DockerBuildException, \
    DockerBuildIOError
from docker_build.utils.argparser import \
    PutAction, \
    parse_key_value_option
from docker_build.daemon import \
    ENV_CONNECTION_TIMEOUT, \
    ENV_IGNORE_CACHE, \
    DEFAULT_DOCKER_CONNECTION_TIMEOUT, \
    DEFAULT_DOCKER_IGNORE_CACHE


def main(argv=None):
    """
    Main function for invoking the Docker Build tool
    """
    # Parse argument list
    parser = argparse.ArgumentParser(
        description="Build tool for creating Docker images"
    )
    parser.add_argument(
        "-a", "--arg",
        dest="build_args",
        type=parse_key_value_option,
        action=PutAction,
        metavar="NAME=VALUE",
        default={},
        help="Passes the value of an argument that is defined in the build file. The arguments "
             "option can be defined multiple times for each argument that is to be populated with "
             "a value. Passes arguments are expected to be Name/Value pairs separated with an "
             "equals sign"
    )
    parser.add_argument(
        "-b", "--build-config-file",
        dest="build_config_file_path",
        type=str,
        default="./docker-build.yml",
        help="The path to the build configuration file that will be used during the build process. "
             "By default the tool will look for the build configuration file in the current "
             "working directory unless overwritten with the use of this option"
    )
    parser.add_argument(
        "-m", "--main-config-file",
        dest="main_config_file_path",
        type=str,
        help="The path to the main configuration file that will be used during the build process. "
             "By default the tool will look for the main configuration file in the users home "
             "directory under the .docker folder. The default behaviour can be overwritten with "
             "the use of this option"
    )
    parser.add_argument(
        "-t", "--tag",
        dest="tag",
        type=str,
        help="If specified the tag value defined in the build file will be overwritten. The tag "
             "is used to commit the final image that is generated by the build tool. Either the "
             "TAG command or the tag option must be specified"
    )
    parser.add_argument(
        "--connection-timeout",
        dest="connection_timeout",
        type=int,
        default=DEFAULT_DOCKER_CONNECTION_TIMEOUT,
        help="The maximum amount of seconds to wait before the connection to the docker daemon "
             "times out. This option can also be set with the use of the environment variable {}"
             .format(ENV_CONNECTION_TIMEOUT)
    )
    parser.add_argument(
        "--ignore-cache",
        dest="ignore_cache",
        action="store_true",
        default=DEFAULT_DOCKER_IGNORE_CACHE,
        help="Determines if the local cache of docker images should be ignored when checking for "
             "the base image that is required by the build. If the option is turned on the remote "
             "repository will always be checked for an updated copy of the image even if it exists "
             "in the local cache. This option can also be set with the use of the environment "
             "variable {}"
             .format(ENV_IGNORE_CACHE)
    )
    parser.add_argument(
        "--keep",
        dest="keep_containers",
        action="store_true",
        default=DEFAULT_KEEP_CONTAINERS,
        help="Keeps the intermediate containers after a build. The default behaviour is to remove "
             "all created containers but sometimes it is useful to leave them for debugging "
             "purposes"
    )
    parser.add_argument(
        "--verbose",
        dest="verbose",
        action="store_true",
        help="Prints more logs during the build process to give more context of what is happening"
             "in during a build of an Docker image"
    )
    parser.add_argument(
        "-v", "--version",
        action="version",
        version="docker-build-tool version {}".format(__version__),
        help="Prints the program's version number and exits"
    )

    try:

        # initialise the logger for the module
        initialise_logging()

        # start the timer to find out at the end how long the build took
        start_time = time.time()

        # parse the command line arguments passed to the tool
        command_line_args = parser.parse_args(argv)

        # determine if lower level logging should be enabled
        if command_line_args.verbose:
            log.setLevel(logging.DEBUG)

        # build the image according to the given build file
        image_id, tag = build(
            command_line_args.build_config_file_path,
            command_line_args.build_args,
            config_file_path=command_line_args.main_config_file_path,
            tag=command_line_args.tag,
            connection_timeout=command_line_args.connection_timeout,
            ignore_cache=command_line_args.ignore_cache,
            keep_containers=command_line_args.keep_containers
        )

        log.info("Created image tagged as {}".format(tag))

        total_build_time = int(time.time() - start_time)
        log.info("Build finished in {} min/s {} sec/s".format(
            int(total_build_time / 60),
            total_build_time % 60
        ))

    except KeyboardInterrupt:
        log.info("Docker Build shutdown by user")
        return 130

    except (DockerBuildException, DockerBuildIOError) as ex:
        log.error("Build failed due to error : {}".format(ex))
        return 1

    except Exception as ex:
        log.exception("Unexpected error during build due to error : {}".format(ex))
        return 1


if __name__ == '__main__':
    sys.exit(main())
