"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsFargateService = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const application_multiple_target_groups_service_base_1 = require("../base/application-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationMultipleTargetGroupsFargateService extends application_multiple_target_groups_service_base_1.ApplicationMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.ApplicationMultipleTargetGroupsFargateService = ApplicationMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,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