from threading import Thread

from easelenium.ui.generator.page_object_generator import PageObjectGenerator
from easelenium.ui.string_utils import StringUtils
from easelenium.ui.utils import FLAG_ALL_AND_EXPAND
from easelenium.ui.widgets.image.image_with_elements import ImageWithElements
from easelenium.ui.widgets.table import Table
from easelenium.ui.widgets.utils import (
    DialogWithText,
    ImageAndTableHelper,
    WxTextCtrlHandler,
    show_dialog,
)
from easelenium.utils import Logger
from wx import (
    EVT_BUTTON,
    EVT_MOTION,
    SP_3D,
    SP_LIVE_UPDATE,
    Button,
    GridBagSizer,
    Panel,
    SplitterWindow,
)
from wx.grid import EVT_GRID_SELECT_CELL


class SelectorFinderTab(Panel):
    def __init__(self, parent):
        Panel.__init__(self, parent)

        self.main_frame = self.GetTopLevelParent()
        self.po_fields = None

        self.__create_widgets()

    def __create_widgets(self):
        sizer = GridBagSizer(5, 5)

        row = 0
        col = 1
        self.bth_reload_img = Button(self, label="Reload image")
        self.bth_reload_img.Bind(EVT_BUTTON, self.__load_img)
        sizer.Add(self.bth_reload_img, pos=(row, col), flag=FLAG_ALL_AND_EXPAND)

        col += 1
        self.bth_reload_selectors = Button(self, label="Find selectors")
        self.bth_reload_selectors.Bind(EVT_BUTTON, self.__find_selectors)
        sizer.Add(self.bth_reload_selectors, pos=(row, col), flag=FLAG_ALL_AND_EXPAND)
        # third row
        row += 1
        col = 0

        splitter = SplitterWindow(self, style=SP_3D | SP_LIVE_UPDATE)

        self.image_panel = ImageWithElements(splitter)
        self.image_panel.static_bitmap.Bind(EVT_MOTION, self.__on_mouse_move)

        self.table = Table(splitter)
        self.table.Bind(EVT_GRID_SELECT_CELL, self.__on_cell_select)

        splitter.SplitHorizontally(self.image_panel, self.table)
        sizer.Add(splitter, pos=(row, col), span=(1, 3), flag=FLAG_ALL_AND_EXPAND)

        sizer.AddGrowableCol(0, 1)
        sizer.AddGrowableRow(row, 1)

        self.SetSizer(sizer)

    def __update_table(self):
        if self.po_fields:
            self.table.load_data(self.po_fields)

    def __on_mouse_move(self, evt):
        ImageAndTableHelper.select_field_on_mouse_move(
            evt, self.po_fields, self.image_panel, self.table
        )

    def __on_cell_select(self, evt):
        self.table.selected_row = evt.GetRow()
        self.image_panel.draw_selected_field(self.table.get_selected_data(), True)
        evt.Skip()

    def __load_img(self, evt=None):
        browser = self.main_frame.get_browser()
        if browser:
            img_path = browser.save_screenshot(self.main_frame.get_tmp_dir())
            self.image_panel.load_image(img_path)
            self.main_frame.set_url(browser.get_current_url())

    def __find_selectors(self, evt):
        browser = self.main_frame.get_browser()
        if browser:
            url = self.main_frame.get_url()
            if not StringUtils.is_url_correct(url):
                show_dialog(self, "Bad url: %s" % url, "Bad url")
            else:
                dialog = DialogWithText(self, "Finding selectors...")
                handler = WxTextCtrlHandler(dialog.txt_ctrl)
                logger = Logger(log_to_console=False, handler=handler)

                dialog.Show()

                generator = PageObjectGenerator(browser, logger)

                def find_selectors():
                    dialog.btn_ok.Disable()
                    self.po_fields = generator.get_all_po_fields(url, None)
                    logger.info("DONE")
                    self.__update_table()
                    dialog.btn_ok.Enable()

                thread = Thread(target=find_selectors)
                thread.setDaemon(True)
                thread.start()
