from sklearn.pipeline import Pipeline as SklearnPipeline
from sklearn.utils.metaestimators import available_if

__all__ = ["Pipeline"]


def _final_estimator_has(attr):
    """Check that final_estimator has `attr`.
    Used together with `avaliable_if` in `Pipeline`. Copied from sklearn as it is a private method.
    """

    def check(self):
        # raise original `AttributeError` if `attr` does not exist
        getattr(self._final_estimator, attr)
        return True

    return check


class Pipeline(SklearnPipeline):
    """ "
    Pipeline for causal inference.

    It inherits from the sklearn pipeline, with some methods that is useful for causal inference models.
    """

    @available_if(_final_estimator_has("predict_ite"))
    def predict_ite(self, X, **predict_ite_params):
        """Transform the data, and apply `predict_ite` with the final estimator.
        Call `transform` of each transformer in the pipeline. The transformed
        data are finally passed to the final estimator that calls `predict_ite`
        method. Only valid if the final estimator implements `predict_ite`.
        Parameters
        ----------
        X : iterable
            Data to predict on. Must fulfill input requirements of first step
            of the pipeline.
        **predict_ite_params : dict of string -> object
            Parameters to the ``predict_ite`` called at the end of all
            transformations in the pipeline. Note that while this may be
            used to return uncertainties from some models with return_std
            or return_cov, uncertainties that are generated by the
            transformations in the pipeline are not propagated to the
            final estimator.

        Returns
        -------
        treatment_effect : ndarray
            Result of calling `predict_ite` on the final estimator.
        """
        Xt = X
        for _, name, transform in self._iter(with_final=False):
            Xt = transform.transform(Xt)
        return self.steps[-1][1].predict_ite(Xt, **predict_ite_params)
