"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const k8s_patch_1 = require("../lib/k8s-patch");
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'applies a patch to k8s'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'MyPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'myResourceName',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                    'Outputs.awscdkawseksKubectlProviderframeworkonEvent0A650005Arn',
                ],
            },
            ResourceName: 'myResourceName',
            ResourceNamespace: 'default',
            ApplyPatchJson: '{"patch":{"to":"apply"}}',
            RestorePatchJson: '{"restore":{"patch":123}}',
            ClusterName: {
                Ref: 'MyCluster8AD82BF8',
            },
            RoleArn: {
                'Fn::GetAtt': [
                    'MyClusterCreationRoleB5FA4FF3',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'defaults to "strategic" patch type if no patchType is specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'MyPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'myResourceName',
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            PatchType: 'strategic',
        }));
        test.done();
    },
    'uses specified to patch type if specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'jsonPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'jsonPatchResource',
            patchType: k8s_patch_1.PatchType.JSON,
        });
        new k8s_patch_1.KubernetesPatch(stack, 'mergePatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'mergePatchResource',
            patchType: k8s_patch_1.PatchType.MERGE,
        });
        new k8s_patch_1.KubernetesPatch(stack, 'strategicPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'strategicPatchResource',
            patchType: k8s_patch_1.PatchType.STRATEGIC,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'jsonPatchResource',
            PatchType: 'json',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'mergePatchResource',
            PatchType: 'merge',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'strategicPatchResource',
            PatchType: 'strategic',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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