"use strict";
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const asg = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const YAML = require("yaml");
const eks = require("../lib");
const kubectl_provider_1 = require("../lib/kubectl-provider");
const bottlerocket_1 = require("../lib/private/bottlerocket");
const util_1 = require("./util");
/* eslint-disable max-len */
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'cluster connections include both control plane and cluster security group'(test) {
        const { stack } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            version: eks.KubernetesVersion.V1_17,
        });
        test.deepEqual(cluster.connections.securityGroups.map(sg => stack.resolve(sg.securityGroupId)), [
            { 'Fn::GetAtt': ['Cluster9EE0221C', 'ClusterSecurityGroupId'] },
            { 'Fn::GetAtt': ['ClusterControlPlaneSecurityGroupD274242C', 'GroupId'] },
        ]);
        test.done();
    },
    'can declare a security group from a different stack'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id);
                this.eksCluster = new eks.Cluster(this, 'Cluster', {
                    version: eks.KubernetesVersion.V1_17,
                    securityGroup: props.sg,
                    vpc: props.vpc,
                });
            }
        }
        class NetworkStack extends cdk.Stack {
            constructor(scope, id) {
                super(scope, id);
                this.vpc = new ec2.Vpc(this, 'Vpc');
                this.securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.vpc });
            }
        }
        const { app } = util_1.testFixture();
        const networkStack = new NetworkStack(app, 'NetworkStack');
        new ClusterStack(app, 'ClusterStack', { sg: networkStack.securityGroup, vpc: networkStack.vpc });
        // make sure we can synth (no circular dependencies between the stacks)
        app.synth();
        test.done();
    },
    'can declare a manifest with a token from a different stack than the cluster that depends on the cluster stack'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.eksCluster = new eks.Cluster(this, 'Cluster', {
                    version: eks.KubernetesVersion.V1_17,
                });
            }
        }
        class ManifestStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                // this role creates a dependency between this stack and the cluster stack
                const role = new iam.Role(this, 'CrossRole', {
                    assumedBy: new iam.ServicePrincipal('sqs.amazonaws.com'),
                    roleName: props.cluster.clusterArn,
                });
                // make sure this manifest doesn't create a dependency between the cluster stack
                // and this stack
                new eks.KubernetesManifest(this, 'cross-stack', {
                    manifest: [{
                            kind: 'ConfigMap',
                            apiVersion: 'v1',
                            metadata: {
                                name: 'config-map',
                            },
                            data: {
                                foo: role.roleArn,
                            },
                        }],
                    cluster: props.cluster,
                });
            }
        }
        const { app } = util_1.testFixture();
        const clusterStack = new ClusterStack(app, 'ClusterStack');
        new ManifestStack(app, 'ManifestStack', { cluster: clusterStack.eksCluster });
        // make sure we can synth (no circular dependencies between the stacks)
        app.synth();
        test.done();
    },
    'can declare a chart with a token from a different stack than the cluster that depends on the cluster stack'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.eksCluster = new eks.Cluster(this, 'Cluster', {
                    version: eks.KubernetesVersion.V1_17,
                });
            }
        }
        class ChartStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                // this role creates a dependency between this stack and the cluster stack
                const role = new iam.Role(this, 'CrossRole', {
                    assumedBy: new iam.ServicePrincipal('sqs.amazonaws.com'),
                    roleName: props.cluster.clusterArn,
                });
                // make sure this chart doesn't create a dependency between the cluster stack
                // and this stack
                new eks.HelmChart(this, 'cross-stack', {
                    chart: role.roleArn,
                    cluster: props.cluster,
                });
            }
        }
        const { app } = util_1.testFixture();
        const clusterStack = new ClusterStack(app, 'ClusterStack');
        new ChartStack(app, 'ChartStack', { cluster: clusterStack.eksCluster });
        // make sure we can synth (no circular dependencies between the stacks)
        app.synth();
        test.done();
    },
    'can declare a HelmChart in a different stack than the cluster'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.eksCluster = new eks.Cluster(this, 'Cluster', {
                    version: eks.KubernetesVersion.V1_17,
                });
            }
        }
        class ChartStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                const resource = new cdk.CfnResource(this, 'resource', { type: 'MyType' });
                new eks.HelmChart(this, `chart-${id}`, { cluster: props.cluster, chart: resource.ref });
            }
        }
        const { app } = util_1.testFixture();
        const clusterStack = new ClusterStack(app, 'ClusterStack');
        new ChartStack(app, 'ChartStack', { cluster: clusterStack.eksCluster });
        // make sure we can synth (no circular dependencies between the stacks)
        app.synth();
        test.done();
    },
    'throws when declaring an ASG role in a different stack than the cluster'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.eksCluster = new eks.Cluster(this, 'Cluster', {
                    version: eks.KubernetesVersion.V1_17,
                });
            }
        }
        class CapacityStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                // the role is create in this stack implicitly by the ASG
                this.group = new asg.AutoScalingGroup(this, 'autoScaling', {
                    instanceType: new ec2.InstanceType('t3.medium'),
                    vpc: props.cluster.vpc,
                    machineImage: new eks.EksOptimizedImage({
                        kubernetesVersion: eks.KubernetesVersion.V1_16.version,
                        nodeType: eks.NodeType.STANDARD,
                    }),
                });
            }
        }
        const { app } = util_1.testFixture();
        const clusterStack = new ClusterStack(app, 'ClusterStack');
        const capacityStack = new CapacityStack(app, 'CapacityStack', { cluster: clusterStack.eksCluster });
        try {
            clusterStack.eksCluster.addAutoScalingGroup(capacityStack.group, {});
            test.ok(false, 'expected error');
        }
        catch (err) {
            test.equal(err.message, 'CapacityStackautoScalingInstanceRoleF041EB53 should be defined in the scope of the ClusterStack stack to prevent circular dependencies');
        }
        test.done();
    },
    'can declare a ServiceAccount in a different stack than the cluster'(test) {
        class ClusterStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.eksCluster = new eks.Cluster(this, 'EKSCluster', {
                    version: eks.KubernetesVersion.V1_17,
                });
            }
        }
        class AppStack extends cdk.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                new eks.ServiceAccount(this, 'testAccount', { cluster: props.cluster, name: 'test-account', namespace: 'test' });
            }
        }
        const { app } = util_1.testFixture();
        const clusterStack = new ClusterStack(app, 'EKSCluster');
        new AppStack(app, 'KubeApp', { cluster: clusterStack.eksCluster });
        // make sure we can synth (no circular dependencies between the stacks)
        app.synth();
        test.done();
    },
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::AWSCDK-EKS-Cluster', {
            Config: {
                roleArn: { 'Fn::GetAtt': ['ClusterRoleFA261979', 'Arn'] },
                version: '1.16',
                resourcesVpcConfig: {
                    securityGroupIds: [{ 'Fn::GetAtt': ['ClusterControlPlaneSecurityGroupD274242C', 'GroupId'] }],
                    subnetIds: [
                        { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                        { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            },
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'us-east-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        kubectl_provider_1.getOrCreateKubectlLayer(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-kubectl',
            },
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region in china'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'cn-north-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        kubectl_provider_1.getOrCreateKubectlLayer(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws-cn:serverlessrepo:cn-north-1:487369736442:applications/lambda-layer-kubectl',
            },
        }));
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                InstanceTypes: [
                    'm5.large',
                ],
                ScalingConfig: {
                    DesiredSize: 2,
                    MaxSize: 2,
                    MinSize: 2,
                },
            }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
                version: CLUSTER_VERSION,
            });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                ScalingConfig: {
                    DesiredSize: 10,
                    MaxSize: 10,
                    MinSize: 10,
                },
            }));
            // expect(stack).to(haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        },
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Private' },
                { Key: 'aws-cdk:subnet-type', Value: 'Private' },
                { Key: 'kubernetes.io/role/internal-elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PrivateSubnet1' },
            ],
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Public' },
                { Key: 'aws-cdk:subnet-type', Value: 'Public' },
                { Key: 'kubernetes.io/role/elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PublicSubnet1' },
            ],
        }));
        test.done();
    },
    'adding capacity creates an ASG without a rolling update policy'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        test.deepEqual(assert_1.expect(stack).value.Resources.ClusterASG0E4BA723.UpdatePolicy, { AutoScalingScheduledAction: { IgnoreUnmodifiedGroupSizeProperties: true } });
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'Cluster9EE0221C' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Default',
                },
            ],
        }));
        test.done();
    },
    'create nodegroup with existing role'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        const cluster = new eks.Cluster(stack, 'cluster', {
            defaultCapacity: 10,
            defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
            version: CLUSTER_VERSION,
        });
        const existingRole = new iam.Role(stack, 'ExistingRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            nodeRole: existingRole,
        });
        // THEN
        test.ok(cluster.defaultNodegroup);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            ScalingConfig: {
                DesiredSize: 10,
                MaxSize: 10,
                MinSize: 10,
            },
        }));
        test.done();
    },
    'adding bottlerocket capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'Cluster9EE0221C' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Bottlerocket',
                },
            ],
        }));
        test.done();
    },
    'adding bottlerocket capacity with bootstrapOptions throws error'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
            bootstrapOptions: {},
        }), /bootstrapOptions is not supported for Bottlerocket/);
        test.done();
    },
    'import cluster with new kubectl private subnets'(test) {
        var _a, _b;
        const { stack, vpc } = util_1.testFixture();
        const cluster = eks.Cluster.fromClusterAttributes(stack, 'Cluster', {
            clusterName: 'cluster',
            kubectlPrivateSubnetIds: vpc.privateSubnets.map(s => s.subnetId),
        });
        test.deepEqual((_a = cluster.kubectlPrivateSubnets) === null || _a === void 0 ? void 0 : _a.map(s => stack.resolve(s.subnetId)), [
            { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
            { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
        ]);
        test.deepEqual((_b = cluster.kubectlPrivateSubnets) === null || _b === void 0 ? void 0 : _b.map(s => s.node.id), [
            'KubectlSubnet0',
            'KubectlSubnet1',
        ]);
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', {
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroupIds: cluster.connections.securityGroups.map(x => x.securityGroupId),
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData,
            clusterSecurityGroupId: cluster.clusterSecurityGroupId,
            clusterEncryptionConfigKeyArn: cluster.clusterEncryptionConfigKeyArn,
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':eks:us-east-1:',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':cluster/',
                                {
                                    'Fn::ImportValue': 'Stack:ExportsOutputRefCluster9EE0221C4853B4C3',
                                },
                            ],
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'mastersRole can be used to map an IAM role to "system:masters"'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        new eks.Cluster(stack, 'Cluster', {
            vpc,
            mastersRole: role,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addManifest can be used to apply k8s manifests on this cluster'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addManifest('manifest1', { foo: 123 });
        cluster.addManifest('manifest2', { bar: 123 }, { boor: [1, 2, 3] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: '[{"foo":123}]',
        }));
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: '[{"bar":123},{"boor":[1,2,3]}]',
        }));
        test.done();
    },
    'kubectl resources can be created in a separate stack'(test) {
        // GIVEN
        const { stack, app } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION }); // cluster is under stack2
        // WHEN resource is under stack2
        const stack2 = new cdk.Stack(app, 'stack2', { env: { account: stack.account, region: stack.region } });
        new eks.KubernetesManifest(stack2, 'myresource', {
            cluster,
            manifest: [{ foo: 'bar' }],
        });
        // THEN
        app.synth(); // no cyclic dependency (see https://github.com/aws/aws-cdk/issues/7231)
        // expect a single resource in the 2nd stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                myresource49C6D325: {
                    Type: 'Custom::AWSCDK-EKS-KubernetesResource',
                    Properties: {
                        ServiceToken: {
                            'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttawscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6BOutputsStackawscdkawseksKubectlProviderframeworkonEvent8897FD9BArn49BEF20C',
                        },
                        Manifest: '[{\"foo\":\"bar\"}]',
                        ClusterName: { 'Fn::ImportValue': 'Stack:ExportsOutputRefclusterC5B25D0D98D553F5' },
                        RoleArn: { 'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttclusterCreationRole2B3B5002ArnF05122FC' },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'adding capacity will automatically map its IAM role'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterdefaultInstanceRoleF20A29CD',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['ClusterMastersRole9AA35625', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['ClusterMastersRole9AA35625', 'Arn'] }]] } },
            });
            test.done();
        },
        'if masters role is defined, it should be included in the config command'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                outputConfigCommand: false,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'Cluster9EE0221C' } },
            });
            test.done();
        },
        '`outputMastersRoleArn` can be used to synthesize an output with the arn of the masters role if defined'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputMastersRoleArn: true,
                mastersRole: new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() }),
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterMastersRoleArnB15964B1: { Value: { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] } },
            });
            test.done();
        },
        'boostrap user-data': {
            'rendered by default for ASGs'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'not rendered if bootstrap is disabled'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapEnabled: false,
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': '#!/bin/bash' });
                test.done();
            },
            // cursory test for options: see test.user-data.ts for full suite
            'bootstrap options'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapOptions: {
                        kubeletExtraArgs: '--node-labels FOO=42',
                    },
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'spot instances': {
                'nodes labeled an tainted accordingly'(test) {
                    // GIVEN
                    const { app, stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    const template = app.synth().getStackByName(stack.stackName).template;
                    const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                    test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                    test.done();
                },
                'interrupt handler is added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, {
                        Release: 'stackclusterchartspotinterrupthandlerdec62e07',
                        Chart: 'aws-node-termination-handler',
                        Values: '{\"nodeSelector.lifecycle\":\"Ec2Spot\"}',
                        Namespace: 'kube-system',
                        Repository: 'https://aws.github.io/eks-charts',
                    }));
                    test.done();
                },
                'its possible to add two capacities with spot instances and only one stop handler will be installed'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('Spot1', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    cluster.addCapacity('Spot2', {
                        instanceType: new ec2.InstanceType('m4.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.countResources(eks.HelmChart.RESOURCE_TYPE, 1));
                    test.done();
                },
            },
        },
        'if bootstrap is disabled cannot specify options'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.throws(() => cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false,
                bootstrapOptions: { awsApiRetryAttempts: 10 },
            }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
            test.done();
        },
        'EksOptimizedImage() with no nodeType always uses STANDARD with LATEST_KUBERNETES_VERSION'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const LATEST_KUBERNETES_VERSION = '1.14';
            // WHEN
            new eks.EksOptimizedImage().getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes(LATEST_KUBERNETES_VERSION)), 'LATEST_KUBERNETES_VERSION should be in ssm parameters');
            test.done();
        },
        'EksOptimizedImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.EksOptimizedImage({ kubernetesVersion: '1.15' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/1.15/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'default cluster capacity with ARM64 instance type comes with nodegroup with correct AmiType'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 1,
                version: CLUSTER_VERSION,
                defaultCapacityInstance: new ec2.InstanceType('m6g.medium'),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
                AmiType: 'AL2_ARM_64',
            }));
            test.done();
        },
        'addNodegroup with ARM64 instance type comes with nodegroup with correct AmiType'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
                defaultCapacityInstance: new ec2.InstanceType('m6g.medium'),
            }).addNodegroup('ng', {
                instanceType: new ec2.InstanceType('m6g.medium'),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Nodegroup', {
                AmiType: 'AL2_ARM_64',
            }));
            test.done();
        },
        'EKS-Optimized AMI with GPU support when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
            }).addCapacity('GPUCapacity', {
                instanceType: new ec2.InstanceType('g4dn.xlarge'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux-2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
        'EKS-Optimized AMI with ARM64 when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
            }).addCapacity('ARMCapacity', {
                instanceType: new ec2.InstanceType('m6g.medium'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('/amazon-linux-2-arm64/')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
        'BottleRocketImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new bottlerocket_1.BottleRocketImage({ kubernetesVersion: '1.17' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsservicebottlerocketaws') &&
                v.Default.includes('/bottlerocket/')), 'BottleRocket AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsservicebottlerocketaws') &&
                v.Default.includes('/aws-k8s-1.17/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'when using custom resource a creation role & policy is defined'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    name: 'my-cluster-name',
                    roleArn: { 'Fn::GetAtt': ['MyClusterRoleBA20FE72', 'Arn'] },
                    version: '1.16',
                    resourcesVpcConfig: {
                        securityGroupIds: [
                            { 'Fn::GetAtt': ['MyClusterControlPlaneSecurityGroup6B658F79', 'GroupId'] },
                        ],
                        subnetIds: [
                            { Ref: 'MyClusterDefaultVpcPublicSubnet1SubnetFAE5A9B6' },
                            { Ref: 'MyClusterDefaultVpcPublicSubnet2SubnetF6D028A0' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet1SubnetE1D0DCDB' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet2Subnet11FEA8D0' },
                        ],
                        endpointPrivateAccess: true,
                        endpointPublicAccess: true,
                    },
                },
            }));
            // role can be assumed by 3 lambda handlers (2 for the cluster resource and 1 for the kubernetes resource)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root'],
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            // policy allows creation role to pass the cluster role and to interact with the cluster (given we know the explicit cluster name)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: [{
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name',
                                        ],
                                    ],
                                }, {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name/*',
                                        ],
                                    ],
                                }],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':eks:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':fargateprofile/my-cluster-name/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'ec2:DescribeVpcs',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':vpc/',
                                        {
                                            Ref: 'MyClusterDefaultVpc76C24A38',
                                        },
                                    ],
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if an explicit cluster name is not provided, the creation role policy is wider (allows interacting with all clusters)'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: ['*'],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'ec2:DescribeVpcs',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':vpc/',
                                        {
                                            Ref: 'MyClusterDefaultVpc76C24A38',
                                        },
                                    ],
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if helm charts are used, the provider role is allowed to assume the creation role'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // WHEN
            cluster.addChart('MyChart', {
                chart: 'foo',
            });
            // THEN
            const providerStack = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(providerStack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'eks:DescribeCluster',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'referencetoStackMyClusterD33CAEABArn',
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'referencetoStackMyClusterCreationRoleA67486E4Arn',
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'HandlerServiceRoleDefaultPolicyCBD0CC91',
                Roles: [
                    {
                        Ref: 'HandlerServiceRoleFCDC14AE',
                    },
                ],
            }));
            test.done();
        },
        'coreDnsComputeType will patch the coreDNS configuration to use a "fargate" compute type and restore to "ec2" upon removal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                coreDnsComputeType: eks.CoreDnsComputeType.FARGATE,
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
                ResourceName: 'deployment/coredns',
                ResourceNamespace: 'kube-system',
                ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
                RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
                ClusterName: {
                    Ref: 'MyCluster8AD82BF8',
                },
                RoleArn: {
                    'Fn::GetAtt': [
                        'MyClusterCreationRoleB5FA4FF3',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'if openIDConnectProvider a new OpenIDConnectProvider resource is created and exposed'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            const provider = cluster.openIdConnectProvider;
            // THEN
            test.equal(provider, cluster.openIdConnectProvider, 'openIdConnect provider is different and created more than once.');
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDKOpenIdConnectProvider', {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'CustomAWSCDKOpenIdConnectProviderCustomResourceProviderHandlerF2C543E0',
                        'Arn',
                    ],
                },
                ClientIDList: [
                    'sts.amazonaws.com',
                ],
                ThumbprintList: [
                    '9e99a48a9960b14926bb7f3b02e22da2b0ab7280',
                ],
                Url: {
                    'Fn::GetAtt': [
                        'Cluster9EE0221C',
                        'OpenIdConnectIssuerUrl',
                    ],
                },
            }));
            test.done();
        },
        'inference instances are supported'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            cluster.addCapacity('InferenceInstances', {
                instanceType: new ec2.InstanceType('inf1.2xlarge'),
                minCapacity: 1,
            });
            const fileContents = fs.readFileSync(path.join(__dirname, '../lib', 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesManifest.RESOURCE_TYPE, {
                Manifest: JSON.stringify([sanitized]),
            }));
            test.done();
        },
        'kubectl resources are always created after all fargate profiles'(test) {
            // GIVEN
            const { stack, app } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // WHEN
            cluster.addFargateProfile('profile1', { selectors: [{ namespace: 'profile1' }] });
            cluster.addManifest('resource1', { foo: 123 });
            cluster.addFargateProfile('profile2', { selectors: [{ namespace: 'profile2' }] });
            new eks.HelmChart(stack, 'chart', { cluster, chart: 'mychart' });
            cluster.addFargateProfile('profile3', { selectors: [{ namespace: 'profile3' }] });
            new eks.KubernetesPatch(stack, 'patch1', {
                cluster,
                applyPatch: { foo: 123 },
                restorePatch: { bar: 123 },
                resourceName: 'foo/bar',
            });
            cluster.addFargateProfile('profile4', { selectors: [{ namespace: 'profile4' }] });
            // THEN
            const template = app.synth().getStackArtifact(stack.artifactId).template;
            const barrier = template.Resources.ClusterKubectlReadyBarrier200052AF;
            test.deepEqual(barrier.DependsOn, [
                'Clusterfargateprofileprofile1PodExecutionRoleE85F87B5',
                'Clusterfargateprofileprofile129AEA3C6',
                'Clusterfargateprofileprofile2PodExecutionRole22670AF8',
                'Clusterfargateprofileprofile233B9A117',
                'Clusterfargateprofileprofile3PodExecutionRole475C0D8F',
                'Clusterfargateprofileprofile3D06F3076',
                'Clusterfargateprofileprofile4PodExecutionRole086057FB',
                'Clusterfargateprofileprofile4A0E3BBE8',
                'ClusterCreationRoleDefaultPolicyE8BDFC7B',
                'ClusterCreationRole360249B6',
                'Cluster9EE0221C',
            ]);
            const kubectlResources = ['chartF2447AFC', 'patch1B964AC93', 'Clustermanifestresource10B1C9505', 'ClusterAwsAuthmanifestFE51F8AE'];
            // check that all kubectl resources depend on the barrier
            for (const r of kubectlResources) {
                test.deepEqual(template.Resources[r].DependsOn, ['ClusterKubectlReadyBarrier200052AF']);
            }
            test.done();
        },
        'kubectl provider role can assume creation role'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            const c1 = new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION });
            // WHEN
            // activate kubectl provider
            c1.addManifest('c1a', { foo: 123 });
            c1.addManifest('c1b', { foo: 123 });
            // THEN
            const providerStack = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(providerStack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'eks:DescribeCluster',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'referencetoStackCluster18DFEAC17Arn',
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'referencetoStackCluster1CreationRoleEF7C9BBCArn',
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'kubectl provider passes environment to lambda'(test) {
        const { stack } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster1', {
            version: CLUSTER_VERSION,
            endpointAccess: eks.EndpointAccess.PRIVATE,
            kubectlEnvironment: {
                Foo: 'Bar',
            },
        });
        cluster.addManifest('resource', {
            kind: 'ConfigMap',
            apiVersion: 'v1',
            data: {
                hello: 'world',
            },
            metadata: {
                name: 'config-map',
            },
        });
        // the kubectl provider is inside a nested stack.
        const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
        assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    Foo: 'Bar',
                },
            },
        }));
        test.done();
    },
    'endpoint access': {
        'public restricted'(test) {
            test.throws(() => {
                eks.EndpointAccess.PUBLIC.onlyFrom('1.2.3.4/32');
            }, /Cannot restric public access to endpoint when private access is disabled. Use PUBLIC_AND_PRIVATE.onlyFrom\(\) instead./);
            test.done();
        },
        'public non restricted without private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PUBLIC,
                vpcSubnets: [{ subnetType: ec2.SubnetType.PUBLIC }],
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // we don't attach vpc config in case endpoint is public only, regardless of whether
            // the vpc has private subnets or not.
            test.equal(template.Resources.Handler886CB40B.Properties.VpcConfig, undefined);
            test.done();
        },
        'public non restricted with private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PUBLIC,
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // we don't attach vpc config in case endpoint is public only, regardless of whether
            // the vpc has private subnets or not.
            test.equal(template.Resources.Handler886CB40B.Properties.VpcConfig, undefined);
            test.done();
        },
        'private without private subnets'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    version: CLUSTER_VERSION,
                    endpointAccess: eks.EndpointAccess.PRIVATE,
                    vpcSubnets: [{ subnetType: ec2.SubnetType.PUBLIC }],
                });
            }, /Vpc must contain private subnets when public endpoint access is disabled/);
            test.done();
        },
        'private with private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // handler should have vpc config
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds.length !== 0);
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SecurityGroupIds.length !== 0);
            test.done();
        },
        'private and non restricted public without private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PUBLIC_AND_PRIVATE,
                vpcSubnets: [{ subnetType: ec2.SubnetType.PUBLIC }],
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // we don't have private subnets, but we don't need them since public access
            // is not restricted.
            test.equal(template.Resources.Handler886CB40B.Properties.VpcConfig, undefined);
            test.done();
        },
        'private and non restricted public with private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PUBLIC_AND_PRIVATE,
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // we have private subnets so we should use them.
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds.length !== 0);
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SecurityGroupIds.length !== 0);
            test.done();
        },
        'private and restricted public without private subnets'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    version: CLUSTER_VERSION,
                    endpointAccess: eks.EndpointAccess.PUBLIC_AND_PRIVATE.onlyFrom('1.2.3.4/32'),
                    vpcSubnets: [{ subnetType: ec2.SubnetType.PUBLIC }],
                });
            }, /Vpc must contain private subnets when public endpoint access is restricted/);
            test.done();
        },
        'private and restricted public with private subnets'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PUBLIC_AND_PRIVATE.onlyFrom('1.2.3.4/32'),
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            // we have private subnets so we should use them.
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds.length !== 0);
            test.ok(template.Resources.Handler886CB40B.Properties.VpcConfig.SecurityGroupIds.length !== 0);
            test.done();
        },
        'private endpoint access selects only private subnets from looked up vpc'(test) {
            const vpcId = 'vpc-12345';
            // can't use the regular fixture because it also adds a VPC to the stack, which prevents
            // us from setting context.
            const stack = new cdk.Stack(new cdk.App(), 'Stack', {
                env: {
                    account: '11112222',
                    region: 'us-east-1',
                },
            });
            stack.node.setContext(`vpc-provider:account=${stack.account}:filter.vpc-id=${vpcId}:region=${stack.region}:returnAsymmetricSubnets=true`, {
                vpcId: vpcId,
                vpcCidrBlock: '10.0.0.0/16',
                subnetGroups: [
                    {
                        name: 'Private',
                        type: 'Private',
                        subnets: [
                            {
                                subnetId: 'subnet-private-in-us-east-1a',
                                cidr: '10.0.1.0/24',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rtb-06068e4c4049921ef',
                            },
                        ],
                    },
                    {
                        name: 'Public',
                        type: 'Public',
                        subnets: [
                            {
                                subnetId: 'subnet-public-in-us-east-1c',
                                cidr: '10.0.0.0/24',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rtb-0ff08e62195198dbb',
                            },
                        ],
                    },
                ],
            });
            const vpc = ec2.Vpc.fromLookup(stack, 'Vpc', {
                vpcId: vpcId,
            });
            new eks.Cluster(stack, 'Cluster', {
                vpc,
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            test.deepEqual(template.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds, [
                'subnet-private-in-us-east-1a',
            ]);
            test.done();
        },
        'private endpoint access considers specific subnet selection'(test) {
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
                vpcSubnets: [{
                        subnets: [ec2.PrivateSubnet.fromSubnetAttributes(stack, 'Private1', {
                                subnetId: 'subnet1',
                                availabilityZone: 'us-east-1a',
                            })],
                    }],
            });
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            const template = assert_1.expect(nested).value;
            test.deepEqual(template.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds, [
                'subnet1',
            ]);
            test.done();
        },
        'can configure private endpoint access'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION, endpointAccess: eks.EndpointAccess.PRIVATE });
            test.equal(assert_1.expect(stack).value.Resources.Cluster1B02DD5A2.Properties.Config.resourcesVpcConfig.endpointPrivateAccess, true);
            test.equal(assert_1.expect(stack).value.Resources.Cluster1B02DD5A2.Properties.Config.resourcesVpcConfig.endpointPublicAccess, false);
            test.done();
        },
        'kubectl provider chooses only private subnets'(test) {
            const { stack } = util_1.testFixture();
            const vpc = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration: [
                    {
                        subnetType: ec2.SubnetType.PRIVATE,
                        name: 'Private1',
                    },
                    {
                        subnetType: ec2.SubnetType.PUBLIC,
                        name: 'Public1',
                    },
                ],
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc,
            });
            cluster.addManifest('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        {
                            Ref: 'referencetoStackCluster18DFEAC17ClusterSecurityGroupId',
                        },
                    ],
                    SubnetIds: [
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet1Subnet6764A0F6Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet2SubnetDFD49645Ref',
                        },
                    ],
                },
            }));
            test.done();
        },
        'kubectl provider limits number of subnets to 16'(test) {
            const { stack } = util_1.testFixture();
            const subnetConfiguration = [];
            for (let i = 0; i < 20; i++) {
                subnetConfiguration.push({
                    subnetType: ec2.SubnetType.PRIVATE,
                    name: `Private${i}`,
                });
            }
            subnetConfiguration.push({
                subnetType: ec2.SubnetType.PUBLIC,
                name: 'Public1',
            });
            const vpc2 = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration,
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc: vpc2,
            });
            cluster.addManifest('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            test.equal(16, assert_1.expect(nested).value.Resources.Handler886CB40B.Properties.VpcConfig.SubnetIds.length);
            test.done();
        },
        'kubectl provider considers vpc subnet selection'(test) {
            const { stack } = util_1.testFixture();
            const subnetConfiguration = [];
            for (let i = 0; i < 20; i++) {
                subnetConfiguration.push({
                    subnetType: ec2.SubnetType.PRIVATE,
                    name: `Private${i}`,
                });
            }
            subnetConfiguration.push({
                subnetType: ec2.SubnetType.PUBLIC,
                name: 'Public1',
            });
            const vpc2 = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 2,
                natGateways: 1,
                subnetConfiguration,
            });
            const cluster = new eks.Cluster(stack, 'Cluster1', {
                version: CLUSTER_VERSION,
                endpointAccess: eks.EndpointAccess.PRIVATE,
                vpc: vpc2,
                vpcSubnets: [{ subnetGroupName: 'Private1' }, { subnetGroupName: 'Private2' }],
            });
            cluster.addManifest('resource', {
                kind: 'ConfigMap',
                apiVersion: 'v1',
                data: {
                    hello: 'world',
                },
                metadata: {
                    name: 'config-map',
                },
            });
            // the kubectl provider is inside a nested stack.
            const nested = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
            assert_1.expect(nested).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        {
                            Ref: 'referencetoStackCluster18DFEAC17ClusterSecurityGroupId',
                        },
                    ],
                    SubnetIds: [
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet1Subnet6764A0F6Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate1Subnet2SubnetDFD49645Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate2Subnet1Subnet586AD392Ref',
                        },
                        {
                            Ref: 'referencetoStackVpcPrivate2Subnet2SubnetE42148C0Ref',
                        },
                    ],
                },
            }));
            test.done();
        },
        'throw when private access is configured without dns support enabled for the VPC'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    vpc: new ec2.Vpc(stack, 'Vpc', {
                        enableDnsSupport: false,
                    }),
                    version: CLUSTER_VERSION,
                });
            }, /Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled/);
            test.done();
        },
        'throw when private access is configured without dns hostnames enabled for the VPC'(test) {
            const { stack } = util_1.testFixture();
            test.throws(() => {
                new eks.Cluster(stack, 'Cluster', {
                    vpc: new ec2.Vpc(stack, 'Vpc', {
                        enableDnsHostnames: false,
                    }),
                    version: CLUSTER_VERSION,
                });
            }, /Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled/);
            test.done();
        },
        'throw when cidrs are configured without public access endpoint'(test) {
            test.throws(() => {
                eks.EndpointAccess.PRIVATE.onlyFrom('1.2.3.4/5');
            }, /CIDR blocks can only be configured when public access is enabled/);
            test.done();
        },
    },
    'getServiceLoadBalancerAddress'(test) {
        const { stack } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION });
        const loadBalancerAddress = cluster.getServiceLoadBalancerAddress('myservice');
        new cdk.CfnOutput(stack, 'LoadBalancerAddress', {
            value: loadBalancerAddress,
        });
        const expectedKubernetesGetId = 'Cluster1myserviceLoadBalancerAddress198CCB03';
        const rawTemplate = assert_1.expect(stack).value;
        // make sure the custom resource is created correctly
        test.deepEqual(rawTemplate.Resources[expectedKubernetesGetId].Properties, {
            ServiceToken: {
                'Fn::GetAtt': [
                    'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                    'Outputs.StackawscdkawseksKubectlProviderframeworkonEvent8897FD9BArn',
                ],
            },
            ClusterName: {
                Ref: 'Cluster1B02DD5A2',
            },
            RoleArn: {
                'Fn::GetAtt': [
                    'Cluster1CreationRoleA231BE8D',
                    'Arn',
                ],
            },
            ObjectType: 'service',
            ObjectName: 'myservice',
            ObjectNamespace: 'default',
            JsonPath: '.status.loadBalancer.ingress[0].hostname',
            TimeoutSeconds: 300,
        });
        // make sure the attribute points to the expected custom resource and extracts the correct attribute
        test.deepEqual(rawTemplate.Outputs.LoadBalancerAddress.Value, { 'Fn::GetAtt': [expectedKubernetesGetId, 'Value'] });
        test.done();
    },
    'custom kubectl layer can be provided'(test) {
        // GIVEN
        const { stack } = util_1.testFixture();
        // WHEN
        const layer = lambda.LayerVersion.fromLayerVersionArn(stack, 'MyLayer', 'arn:of:layer');
        new eks.Cluster(stack, 'Cluster1', {
            version: CLUSTER_VERSION,
            kubectlLayer: layer,
        });
        // THEN
        const providerStack = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
        assert_1.expect(providerStack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Layers: ['arn:of:layer'],
        }));
        test.done();
    },
    'SAR-based kubectl layer can be customized'(test) {
        // GIVEN
        const { stack } = util_1.testFixture();
        // WHEN
        const layer = new eks.KubectlLayer(stack, 'Kubectl', {
            applicationId: 'custom:app:id',
            version: '2.3.4',
        });
        new eks.Cluster(stack, 'Cluster1', {
            version: CLUSTER_VERSION,
            kubectlLayer: layer,
        });
        // THEN
        const providerStack = stack.node.tryFindChild('@aws-cdk/aws-eks.KubectlProvider');
        assert_1.expect(providerStack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Layers: [{ Ref: 'referencetoStackKubectl7F29063EOutputsLayerVersionArn' }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'custom:app:id',
                SemanticVersion: '2.3.4',
            },
        }));
        test.done();
    },
    'create a cluster using custom resource with secrets encryption using KMS CMK'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', {
            vpc,
            version: CLUSTER_VERSION,
            secretsEncryptionKey: new kms.Key(stack, 'Key'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::AWSCDK-EKS-Cluster', {
            Config: {
                encryptionConfig: [{
                        provider: {
                            keyArn: {
                                'Fn::GetAtt': [
                                    'Key961B73FD',
                                    'Arn',
                                ],
                            },
                        },
                        resources: ['secrets'],
                    }],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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