"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrCreateKubectlLayer = exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const cluster_1 = require("./cluster");
const kubectl_layer_1 = require("./kubectl-layer");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const layer = (_a = cluster.kubectlLayer) !== null && _a !== void 0 ? _a : getOrCreateKubectlLayer(this);
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            layers: [layer],
            memorySize: 256,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.handlerRole = handler.role;
        this.handlerRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${cluster.node.uniqueId}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
/**
 * Gets or create a singleton instance of KubectlLayer.
 *
 * (exported for unit tests).
 */
function getOrCreateKubectlLayer(scope, props = {}) {
    const stack = core_1.Stack.of(scope);
    const id = 'kubectl-layer-' + (props.version ? props.version : '8C2542BC-BF2B-4DFE-B765-E181FD30A9A0');
    const exists = stack.node.tryFindChild(id);
    if (exists) {
        return exists;
    }
    return new kubectl_layer_1.KubectlLayer(stack, id, props);
}
exports.getOrCreateKubectlLayer = getOrCreateKubectlLayer;
//# sourceMappingURL=data:application/json;base64,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