# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_msc.ipynb.

# %% auto 0
__all__ = ['load_malayalam_speech_corpus_dataset', 'evaluate_whisper_model_msc']

# %% ../nbs/02_msc.ipynb 3
import time
from typing import List

import pandas as pd
from datasets import load_dataset, Audio
from jiwer import wer, cer
from transformers import pipeline
from tqdm.notebook import tqdm

from malayalam_asr_benchmarking.utils import (
    whisper_norm,
    is_target_text_in_range,
    get_text,
    normalise,
    data,
    get_model_size,
    clear_gpu_memory,
)

# %% ../nbs/02_msc.ipynb 5
def load_malayalam_speech_corpus_dataset():
    dataset = load_dataset("thennal/msc", split="train")
    dataset = dataset.cast_column("audio", Audio(sampling_rate=16000))
    dataset = dataset.map(normalise)
    dataset = dataset.filter(is_target_text_in_range, input_columns=["norm_text"])
    return dataset

# %% ../nbs/02_msc.ipynb 6
def evaluate_whisper_model_msc(
    model_name: str,  # The model name
    werlist: List[float],  # WER List
    cerlist: List[float],  # CER list
    modelsizelist: List[str],  # model size list
    timelist: List[float],  # time(s) list
    bs: int = 16,  # batch size
) -> None:
    whisper_asr = pipeline("automatic-speech-recognition", model=model_name, device=0)
    dataset = load_malayalam_speech_corpus_dataset()

    predictions = []
    references = []

    start = time.time()
    print("process of calculating predictions")
    for out in tqdm(whisper_asr(data(dataset), batch_size=bs)):
        predictions.append(whisper_norm(out["text"]))
        references.append(out["reference"][0])

    print("completed getting predictions")
    end = time.time()
    print(f"Total time taken: {end - start}")
    timelist.append(end - start)

    df = pd.DataFrame({"predictions": predictions, "ground_truth": references})
    df["model_name"] = model_name
    df["wer"] = df.apply(
        lambda row: wer(row["ground_truth"], row["predictions"]), axis=1
    )
    df["cer"] = df.apply(
        lambda row: cer(row["ground_truth"], row["predictions"]), axis=1
    )
    df["total_time"] = end - start

    rwer = wer(references, predictions)
    rwer = round(100 * rwer, 2)
    df["total_wer"] = rwer
    werlist.append(rwer)
    print(f"The WER of model: {rwer}")

    rcer = cer(references, predictions)
    rcer = round(100 * rcer, 2)
    df["total_cer"] = rcer
    cerlist.append(rcer)
    print(f"The CER of model: {rcer}")

    print(f"The model size is: {get_model_size(whisper_asr.model)}")
    modelsizelist.append(get_model_size(whisper_asr.model))
    df["model_size"] = get_model_size(whisper_asr.model)

    save_name = model_name.split("/")
    print(save_name)
    df.to_parquet(
        f"/home/malayalam_msc_benchmarking/{save_name[0]}_{save_name[1]}_msc.parquet"
    )

    clear_gpu_memory()
