"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 *
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 *
 * @stability stable
 */
class Template {
    constructor(template) {
        this.template = template;
    }
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     *
     * @param stack the CDK Stack to run assertions on.
     * @stability stable
     */
    static fromStack(stack) {
        return new Template(toTemplate(stack));
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory JSON object.
     *
     * @param template the CloudFormation template formatted as a nested set of records.
     * @stability stable
     */
    static fromJSON(template) {
        return new Template(template);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a JSON string.
     *
     * @param template the CloudFormation template in.
     * @stability stable
     */
    static fromString(template) {
        return new Template(JSON.parse(template));
    }
    /**
     * The CloudFormation template deserialized into an object.
     *
     * @stability stable
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the template.
     *
     * @param type the resource type;
     * @param count number of expected instances.
     * @stability stable
     */
    resourceCountIs(type, count) {
        const counted = resources_1.countResources(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavour, use other matchers in the `Match` class.
     *
     * @param type the resource type;
     * @param props the 'Properties' section of the resource as should be expected in the template.
     * @stability stable
     */
    hasResourceProperties(type, props) {
        const matchError = resources_1.hasResourceProperties(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param type the resource type;
     * @param props the entire defintion of the resource as should be expected in the template.
     * @stability stable
     */
    hasResource(type, props) {
        const matchError = resources_1.hasResource(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     *
     * @param type the type to match in the CloudFormation template.
     * @param props by default, matches all resources with the given type.
     * @stability stable
     */
    findResources(type, props = {}) {
        return resources_1.findResources(this.template, type, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the output.
     * @param props the output as should be expected in the template.
     * @stability stable
     */
    hasOutput(logicalId, props) {
        const matchError = outputs_1.hasOutput(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the output.
     * @param props by default, matches all Outputs in the template.
     * @stability stable
     */
    findOutputs(logicalId, props = {}) {
        return outputs_1.findOutputs(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the mapping.
     * @param props the output as should be expected in the template.
     * @stability stable
     */
    hasMapping(logicalId, props) {
        const matchError = mappings_1.hasMapping(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the mapping.
     * @param props by default, matches all Mappings in the template.
     * @stability stable
     */
    findMappings(logicalId, props = {}) {
        return mappings_1.findMappings(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value.
     *
     * @param expected the expected CloudFormation template as key-value pairs.
     * @stability stable
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
exports.Template = Template;
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "@aws-cdk/assertions.Template", version: "1.135.0" };
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,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