"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const absent_1 = require("./private/matchers/absent");
const type_1 = require("./private/type");
/**
 * Partial and special matching during template assertions.
 *
 * @stability stable
 */
class Match {
    /**
     * Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @stability stable
     */
    static absent() {
        return new absent_1.AbsentMatch('absent');
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @stability stable
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * Matches any string-encoded JSON and applies the specified pattern after parsing it.
     *
     * @param pattern the pattern to match after parsing the encoded JSON.
     * @stability stable
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * Matches any non-null value at the target.
     *
     * @stability stable
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.135.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`,
            });
            return result;
        }
        if (actual !== this.pattern) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected ${this.pattern} but received ${actual}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type array but received ${type_1.getType(actual)}`,
            });
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected array of length ${this.pattern.length} but received ${actual.length}`,
            });
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            const matcherName = matcher.name;
            if (this.subsequence && (matcherName == 'absent' || matcherName == 'anyValue')) {
                // array subsequence matcher is not compatible with anyValue() or absent() matcher. They don't make sense to be used together.
                throw new Error(`The Matcher ${matcherName}() cannot be nested within arrayWith()`);
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Missing element${element}at pattern index ${patternIdx}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type object but received ${type_1.getType(actual)}`,
            });
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.recordFailure({
                        matcher: this,
                        path: [`/${a}`],
                        message: 'Unexpected key',
                    });
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (!(patternKey in actual) && !(patternVal instanceof absent_1.AbsentMatch)) {
                result.recordFailure({
                    matcher: this,
                    path: [`/${patternKey}`],
                    message: 'Missing key',
                });
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected JSON as a string but found ${type_1.getType(actual)}`,
            });
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.recordFailure({
                    matcher: this,
                    path: [],
                    message: `Invalid JSON string: ${actual}`,
                });
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`,
            });
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: 'Expected a value but found none',
            });
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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