# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/01_models.ipynb (unless otherwise specified).

__all__ = ['UNetConvBlock', 'UNetUpBlock', 'UNet2D', 'unet_ronneberger2015', 'unet_falk2019', 'unet_deepflash2',
           'unet_custom', 'FPN', 'UnetBlock', 'ASPP', 'UneXt50', 'unext50_deepflash2', 'load_smp_model',
           'get_default_shapes']

# Cell
import torch
from torch import nn
import torch.nn.functional as F
from torch import Tensor
from fastai.layers import PixelShuffle_ICNR, ConvLayer
from fastcore.utils import store_attr
from torchvision.models.resnet import ResNet, Bottleneck
from .utils import import_package
import urllib

# Cell
class UNetConvBlock(nn.Module):
    def __init__(self, in_size, out_size, padding, batch_norm,
                 dropout=0., neg_slope=0.1):
        super(UNetConvBlock, self).__init__()
        block = []

        if dropout>0.:
            block.append(nn.Dropout(p=dropout))
        block.append(nn.Conv2d(in_size, out_size, kernel_size=3, padding=int(padding)))
        if batch_norm:
            block.append(nn.BatchNorm2d(out_size))
        block.append(nn.LeakyReLU(negative_slope=neg_slope))


        block.append(nn.Conv2d(out_size, out_size, kernel_size=3, padding=int(padding)))
        if batch_norm:
            block.append(nn.BatchNorm2d(out_size))
        block.append(nn.LeakyReLU(negative_slope=neg_slope))

        self.block = nn.Sequential(*block)

    def forward(self, x):
        out = self.block(x)
        return out

# Cell
class UNetUpBlock(nn.Module):
    def __init__(self, in_size, out_size, up_mode, padding, batch_norm,
                 dropout=0., neg_slope=0.1):
        super(UNetUpBlock, self).__init__()
        up_block = []
        if dropout>0.:
            up_block.append(nn.Dropout(p=dropout))
        if up_mode == 'upconv':
            up_block.append(nn.ConvTranspose2d(in_size, out_size, kernel_size=2, stride=2))
        elif up_mode == 'upsample':
            up_block.append(nn.Upsample(mode='bilinear', scale_factor=2, align_corners=True))
            up_block.append(nn.Conv2d(in_size, out_size, kernel_size=1))
        if batch_norm:
            up_block.append(nn.BatchNorm2d(out_size))
        up_block.append(nn.LeakyReLU(negative_slope=neg_slope))

        self.up = nn.Sequential(*up_block)
        self.conv_block = UNetConvBlock(in_size, out_size, padding, batch_norm)

    def center_crop(self, layer, target_size):
        _, _, layer_height, layer_width = layer.size()
        diff_y = (layer_height - target_size[0]) // 2
        diff_x = (layer_width - target_size[1]) // 2
        return layer[
            :, :, diff_y : (diff_y + target_size[0]), diff_x : (diff_x + target_size[1])
        ]

    def forward(self, x, bridge):
        up = self.up(x)
        crop1 = self.center_crop(bridge, up.shape[2:])
        out = torch.cat([up, crop1], 1)
        out = self.conv_block(out)

        return out

# Cell
class UNet2D(nn.Module):
    "Pytorch U-Net Implementation"
    def __init__(
        self,
        in_channels=1,
        n_classes=2,
        depth=5,
        wf=6,
        padding=False,
        batch_norm=False,
        dropout = 0.,
        neg_slope=0.,
        up_mode='upconv',
        **kwargs
    ):

        super().__init__()
        assert up_mode in ('upconv', 'upsample')
        self.padding = padding
        self.depth = depth
        prev_channels = in_channels
        self.down_path = nn.ModuleList()
        for i in range(depth):
            if batch_norm:
                bn = True if i>0 else False
            else:
                bn = False
            if dropout>0.:
                do = dropout if i>2 else 0.
            else:
                do = 0.
            self.down_path.append(
                UNetConvBlock(prev_channels, 2 ** (wf + i), padding,
                              batch_norm=bn, dropout=do,neg_slope=neg_slope)
            )
            prev_channels = 2 ** (wf + i)

        self.up_path = nn.ModuleList()
        for i in reversed(range(depth - 1)):
            if batch_norm:
                bn = True if i>0 else False
            else:
                bn = False
            if dropout>0.:
                do = dropout if i>2 else 0.
            else:
                do = 0.
            self.up_path.append(
                UNetUpBlock(prev_channels, 2 ** (wf + i), up_mode, padding,
                            batch_norm=bn, dropout=do, neg_slope=neg_slope)
            )
            prev_channels = 2 ** (wf + i)

        self.last = nn.Conv2d(prev_channels, n_classes, kernel_size=1)

    def _initialize_weights(self):
        """Initialize layer weights"""
        for m in self.modules():
            if isinstance(m, nn.Conv2d) or isinstance(m, nn.ConvTranspose2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_in', nonlinearity='leaky_relu')

    def forward(self, x):
        blocks = []
        for i, down in enumerate(self.down_path):
            x = down(x)
            if i != len(self.down_path) - 1:
                blocks.append(x)
                x = F.max_pool2d(x, 2)

        for i, up in enumerate(self.up_path):
            x = up(x, blocks[-i - 1])

        return self.last(x)

# Cell
_MODEL_BASE_URL = 'https://github.com/matjesg/deepflash2/releases/download/model_library/'
def _load_pretrained(model, arch, dataset, progress=True):
    "Loads pretrained model weights"
    url = f'{_MODEL_BASE_URL}{dataset}-{arch}.pth'
    try:
        state_dict = torch.hub.load_state_dict_from_url(url, map_location='cpu', progress=progress)
    except:
        print(f"Error: No weights available for model {arch} trained on {dataset}.")
        print(f"Continuing without pretrained weights.")
        return
    try:
        if arch in ["unet_deepflash2",  "unet_falk2019", "unet_ronnberger2015", "unet_custom"]:
            if model.state_dict()['last.weight'].shape != state_dict['last.weight'].shape:
                print(f"No pretrained weights for {model.state_dict()['last.weight'].shape[0]} classes in final layer.")
                state_dict.pop('last.bias')
                state_dict.pop('last.weight')
        elif arch=='unext50_deepflash2':
            if model.state_dict()['final_conv.0.weight'].shape != state_dict['final_conv.0.weight'].shape:
                print(f"No pretrained weights for {model.state_dict()['final_conv.0.weight'].shape[0]} classes in final layer.")
                state_dict.pop('final_conv.0.bias')
                state_dict.pop('final_conv.0.weight')

        # TODO Better handle different number of input channels
        _ = model.load_state_dict(state_dict, strict=False)
        print(f"Loaded model weights trained on {dataset}.")

    except Exception as e:
        print(f'Error: {e}')
        print(f"Continuing without pretrained weights.")

# Cell
def unet_ronneberger2015(in_channels=1 ,n_classes=2, pretrained=None, progress=True, **kwargs):
    "Original U-Net architecture based on Ronnberger et al. (2015)"
    model = UNet2D(in_channels=in_channels, n_classes=n_classes,
                   depth=5, wf=6, padding=False, batch_norm=False,
                   neg_slope=0., up_mode='upconv', dropout=0, **kwargs)
    if pretrained is not None:
        _load_pretrained(model, arch='unet_deepflash2', dataset=pretrained, progress=progress)
    return model

# Cell
def unet_falk2019(in_channels=1 ,n_classes=2, pretrained=None, progress=True, **kwargs):
    "U-Net architecture based on Falk et al. (2019)"
    model = UNet2D(in_channels=in_channels, n_classes=n_classes,
               depth=5, wf=6, padding=False, batch_norm=False,
               neg_slope=0.1, up_mode='upconv', dropout=0, **kwargs)
    if pretrained is not None:
        _load_pretrained(model, arch='unet_deepflash2', dataset=pretrained, progress=progress)
    return model

# Cell
def unet_deepflash2(in_channels=1 ,n_classes=2, pretrained=None, progress=True, dropout=.5, **kwargs):
    "U-Net model optimized for deepflash2"
    model = UNet2D(in_channels=in_channels, n_classes=n_classes, dropout=dropout,
                   depth=5, wf=6, padding=False, batch_norm=True,
                   neg_slope=0.1, up_mode='upconv', **kwargs)
    if pretrained is not None:
        _load_pretrained(model, arch='unet_deepflash2', dataset=pretrained, progress=progress)
    return model

# Cell
def unet_custom(in_channels=1 ,n_classes=2, pretrained=None, progress=True, **kwargs):
    "Customizable U-Net model. Customize via kwargs"
    model = UNet2D(in_channels=in_channels, n_classes=n_classes, **kwargs)
    if pretrained:
        print('No pretrained weights available for custom architecture.')
    return model

# Cell
class FPN(nn.Module):
    def __init__(self, input_channels:list, output_channels:list):
        super().__init__()
        self.convs = nn.ModuleList(
            [nn.Sequential(nn.Conv2d(in_ch, out_ch*2, kernel_size=3, padding=0),
             nn.ReLU(inplace=True), nn.BatchNorm2d(out_ch*2),
             nn.Conv2d(out_ch*2, out_ch, kernel_size=3, padding=1))
            for in_ch, out_ch in zip(input_channels, output_channels)])

    def forward(self, xs:list, last_layer):
        #hcs = [F.interpolate(c(x),scale_factor=2**(len(self.convs)-i),mode='bilinear', align_corners=True)
        #       for i,(c,x) in enumerate(zip(self.convs, xs))]
        pad_list = [-14,-6,-2,0]
        hcs = [F.pad(F.interpolate(c(x),scale_factor=2**(len(self.convs)-i),mode='bilinear', align_corners=True),(p, p, p, p))
               for i,(c,x,p) in enumerate(zip(self.convs, xs, pad_list))] ## pad last dims by -1 on each side
        hcs.append(last_layer)
        return torch.cat(hcs, dim=1)

# Cell
class UnetBlock(nn.Module):
    def __init__(self, up_in_c:int, x_in_c:int, nf:int=None, blur:bool=False,
                 self_attention:bool=False, padding:int=1, **kwargs):
        super().__init__()
        self.shuf = PixelShuffle_ICNR(up_in_c, up_in_c//2, blur=blur, **kwargs)
        self.bn = nn.BatchNorm2d(x_in_c)
        ni = up_in_c//2 + x_in_c
        nf = nf if nf is not None else max(up_in_c//2,32)
        self.conv1 = ConvLayer(ni, nf, norm_type=None, padding=padding, **kwargs)
        self.conv2 = ConvLayer(nf, nf, norm_type=None, padding=padding,
            xtra=SelfAttention(nf) if self_attention else None, **kwargs)
        self.relu = nn.ReLU(inplace=True)

    def forward(self, up_in:Tensor, left_in:Tensor) -> Tensor:
        s = left_in
        up_out = self.shuf(up_in)
        cat_x = self.relu(torch.cat([up_out, self.bn(s)], dim=1))
        return self.conv2(self.conv1(cat_x))

# Cell
class _ASPPModule(nn.Module):
    def __init__(self, inplanes, planes, kernel_size, padding, dilation, groups=1):
        super().__init__()
        self.atrous_conv = nn.Conv2d(inplanes, planes, kernel_size=kernel_size,
                stride=1, padding=padding, dilation=dilation, bias=False, groups=groups)
        self.bn = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU()

        self._init_weight()

    def forward(self, x):
        x = self.atrous_conv(x)
        x = self.bn(x)

        return self.relu(x)

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

# Cell
class ASPP(nn.Module):
    def __init__(self, inplanes=512, mid_c=256, dilations=[6, 12, 18, 24], out_c=None):
        super().__init__()
        self.aspps = [_ASPPModule(inplanes, mid_c, 1, padding=0, dilation=1)] + \
            [_ASPPModule(inplanes, mid_c, 3, padding=d, dilation=d,groups=4) for d in dilations]
        self.aspps = nn.ModuleList(self.aspps)
        self.global_pool = nn.Sequential(nn.AdaptiveMaxPool2d((1, 1)),
                        nn.Conv2d(inplanes, mid_c, 1, stride=1, bias=False),
                        nn.BatchNorm2d(mid_c), nn.ReLU())
        out_c = out_c if out_c is not None else mid_c
        self.out_conv = nn.Sequential(nn.Conv2d(mid_c*(2+len(dilations)), out_c, 1, bias=False),
                                    nn.BatchNorm2d(out_c), nn.ReLU(inplace=True))
        self.conv1 = nn.Conv2d(mid_c*(2+len(dilations)), out_c, 1, bias=False)
        self._init_weight()

    def forward(self, x):
        x0 = self.global_pool(x)
        xs = [aspp(x) for aspp in self.aspps]
        x0 = F.interpolate(x0, size=xs[0].size()[2:], mode='bilinear', align_corners=True)
        x = torch.cat([x0] + xs, dim=1)
        return self.out_conv(x)

    def _init_weight(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                torch.nn.init.kaiming_normal_(m.weight)
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

# Cell
class UneXt50(nn.Module):
    def __init__(
        self,
        in_channels=1,
        n_classes=2,
        stride=1,
        inplanes=64,
        pre_ssl = True,
        **kwargs):
        super().__init__()
        store_attr('in_channels, n_classes, inplanes, pre_ssl')
        #encoder
        if pre_ssl:
            m = torch.hub.load('facebookresearch/semi-supervised-ImageNet1K-models','resnext50_32x4d_ssl')
        else:
            m = ResNet(Bottleneck, [3, 4, 6, 3], groups=32, width_per_group=4)
        m.conv1.padding = (0,0)

        if in_channels<3:
            #print('Cutting input layer weights to', in_channels, 'channel(s).')
            with torch.no_grad():
                m.conv1.weight = nn.Parameter(m.conv1.weight[:,:in_channels,...])
        elif in_channels>3:
            m.conv1 = nn.Conv2d(1, self.inplanes, kernel_size=7, stride=2, bias=False)

        #self.bn1 =  m.bn1 if in_channels==3 else nn.BatchNorm2d(self.inplanes)
        self.enc0 = nn.Sequential(m.conv1, m.bn1, nn.ReLU(inplace=True))
        self.enc1 = nn.Sequential(nn.MaxPool2d(kernel_size=3, stride=2, padding=1, dilation=1),
                            m.layer1) #256
        self.enc2 = m.layer2 #512
        self.enc3 = m.layer3 #1024
        self.enc4 = m.layer4 #2048
        #aspp with customized dilatations
        self.aspp = ASPP(2048,256,out_c=512,dilations=[stride*1,stride*2,stride*3,stride*4])
        self.drop_aspp = nn.Dropout2d(0.5)
        #decoder
        self.dec4 = UnetBlock(512,1024,256, padding=0)
        self.dec3 = UnetBlock(256,512,128, padding=0)
        self.dec2 = UnetBlock(128,256,64, padding=0)
        self.dec1 = UnetBlock(64,64,32, padding=0)
        self.fpn = FPN([512,256,128,64],[16]*4)
        self.drop = nn.Dropout2d(0.1)
        self.final_conv = ConvLayer(32+16*4, n_classes, ks=1, norm_type=None, act_cls=None)

    def forward(self, x):
        c = [-4,-12,-28] #crop list
        enc0 = self.enc0(x)
        enc1 = self.enc1(enc0)
        enc2 = self.enc2(enc1)
        enc3 = self.enc3(enc2)
        enc4 = self.enc4(enc3)
        enc5 = self.aspp(enc4)
        dec3 = self.dec4(self.drop_aspp(enc5),enc3)
        enc2 = F.pad(enc2, (c[0], c[0], c[0], c[0]))
        dec2 = self.dec3(dec3,enc2)
        enc1 = F.pad(enc1, (c[1],c[1], c[1], c[1]))
        dec1 = self.dec2(dec2,enc1)
        enc0 = F.pad(enc0, (c[2], c[2], c[2], c[2]))
        dec0 = self.dec1(dec1,enc0)
        x = self.fpn([enc5, dec3, dec2, dec1], dec0)
        x = self.final_conv(self.drop(x))
        x = F.interpolate(x,scale_factor=2,mode='bilinear',align_corners=True)
        return x

# Cell
def unext50_deepflash2(in_channels=1 ,n_classes=2, pretrained=None, progress=True, **kwargs):
    "UneXt50 model. Customize via kwargs"
    model = UneXt50(in_channels=in_channels, n_classes=n_classes, **kwargs)
    if pretrained is not None:
        _load_pretrained(model, arch='unext50_deepflash2', dataset=pretrained, progress=progress)
    return model

# Cell
def load_smp_model(arch, **kwargs):
    'Load segmentation_models_pytorch model'
    smp = import_package('segmentation_models_pytorch')
    if arch=="Unet": return smp.Unet(**kwargs)
    elif arch=="UnetPlusPlus": return smp.UnetPlusPlus(**kwargs)
    elif arch=="MAnet":return smp.MAnet(**kwargs)
    elif arch=="FPN": return smp.FPN(**kwargs)
    elif arch=="PAN": return smp.PAN(**kwargs)
    elif arch=="PSPNet": return smp.PSPNet(**kwargs)
    elif arch=="Linknet": return smp.Linknet(**kwargs)
    elif arch=="DeepLabV3": return smp.DeepLabV3(**kwargs)
    elif arch=="DeepLabV3Plus": return smp.DeepLabV3Plus(**kwargs)
    else: raise NotImplementedError

# Cell
def get_default_shapes(arch):
    if arch in ["unet_deepflash2",  "unet_falk2019", "unet_ronnberger2015", "unet_custom"]:
        return {'tile_shape' : (540, 540), 'padding' : (184, 184)}

    elif arch in ["unext50_deepflash2"]:
        return {'tile_shape' : (518, 518), 'padding' : (126, 126)}

    else:
        return {'tile_shape' : (512, 512), 'padding' : (0, 0)}