"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: (_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        },
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or', 'GetAtt',
].map(name => makeTagForCfnIntrinsic(name, true)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'yaml-1.1',
    });
}
//# sourceMappingURL=data:application/json;base64,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