"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const fact_1 = require("./fact");
/**
 * (experimental) Information pertaining to an AWS region.
 *
 * @experimental
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     * may not be an exaustive list of all available AWS regions.
     * @experimental
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * (experimental) Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     * @returns a mapping with AWS region codes as the keys,
     * and the fact in the given region as the value for that key
     * @experimental
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * (experimental) Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1).
     * @experimental
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * (experimental) Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     *
     * @experimental
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * (experimental) The domain name suffix (e.g: amazonaws.com) for this region.
     *
     * @experimental
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * (experimental) The name of the ARN partition for this region (e.g: aws).
     *
     * @experimental
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * (experimental) The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com).
     *
     * @experimental
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * (experimental) The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX).
     *
     * @experimental
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * (experimental) The prefix for VPC Endpoint Service names, cn.com.amazonaws.vpce for China regions, com.amazonaws.vpce otherwise.
     *
     * @experimental
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * (experimental) The name of the service principal for a given service in this region.
     *
     * @param service the service name (e.g: s3.amazonaws.com).
     * @experimental
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * (experimental) The account ID for ELBv2 in this region.
     *
     * @experimental
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * (experimental) The ID of the AWS account that owns the public ECR repository containing the AWS Deep Learning Containers images in this region.
     *
     * @experimental
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
    /**
     * (experimental) The ID of the AWS account that owns the public ECR repository that contains the AWS App Mesh Envoy Proxy images in a given region.
     *
     * @experimental
     */
    get appMeshRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.APPMESH_ECR_ACCOUNT);
    }
}
exports.RegionInfo = RegionInfo;
//# sourceMappingURL=data:application/json;base64,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