"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const crypto = require("crypto");
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * (experimental) A standard synth with a generated buildspec.
 *
 * @experimental
 */
class SimpleSynthAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * (experimental) Create a standard NPM synth action.
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     *
     * @experimental
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Create a standard Yarn synth action.
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     *
     * @experimental
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * (experimental) Project generated to run the synth command.
     *
     * @experimental
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: _util_1.filterEmpty([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                    ]),
                },
                build: {
                    commands: _util_1.filterEmpty([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environment = { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment };
        const environmentVariables = {
            ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
            ...this.props.environmentVariables,
        };
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment,
            buildSpecString: buildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_d = this.props.additionalArtifacts) !== null && _d !== void 0 ? _d : []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * (experimental) The CodeBuild Project's principal.
     *
     * @experimental
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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