"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) A Pipeline to deploy CDK apps.
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 *
 * @experimental
 */
class CdkPipeline extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        if (props.codePipeline) {
            if (props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = props.codePipeline;
        }
        else {
            this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: props.pipelineName,
                crossAccountKeys: props.crossAccountKeys,
                restartExecutionOnUpdate: true,
            });
        }
        if (props.sourceAction && !props.synthAction) {
            // Because of ordering limitations, you can: bring your own Source, bring your own
            // Both, or bring your own Nothing. You cannot bring your own Build (which because of the
            // current CodePipeline API must go BEFORE what we're adding) and then having us add a
            // Source after it. That doesn't make any sense.
            throw new Error('When passing a \'sourceAction\' you must also pass a \'synthAction\' (or a \'codePipeline\' that already has both)');
        }
        if (!props.sourceAction && (!props.codePipeline || props.codePipeline.stages.length < 1)) {
            throw new Error('You must pass a \'sourceAction\' (or a \'codePipeline\' that already has a Source stage)');
        }
        if (props.sourceAction) {
            this._pipeline.addStage({
                stageName: 'Source',
                actions: [props.sourceAction],
            });
        }
        if (props.synthAction) {
            this._pipeline.addStage({
                stageName: 'Build',
                actions: [props.synthAction],
            });
        }
        this._pipeline.addStage({
            stageName: 'UpdatePipeline',
            actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                    cloudAssemblyInput: this._cloudAssemblyArtifact,
                    pipelineStackName: pipelineStack.stackName,
                    cdkCliVersion: props.cdkCliVersion,
                    projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                })],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
        });
        core_1.Aspects.of(this).add({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * (experimental) The underlying CodePipeline object.
     *
     * You can use this to add more Stages to the pipeline, or Actions
     * to Stages.
     *
     * @experimental
     */
    get codePipeline() {
        return this._pipeline;
    }
    /**
     * (experimental) Access one of the pipeline's stages by stage name.
     *
     * You can use this to add more Actions to a stage.
     *
     * @experimental
     */
    stage(stageName) {
        return this._pipeline.stage(stageName);
    }
    /**
     * (experimental) Add pipeline stage that will deploy the given application stage.
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     *
     * @experimental
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * (experimental) Add a new, empty stage to the pipeline.
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     *
     * @experimental
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * (experimental) Get the StackOutput object that holds this CfnOutput's value in this pipeline.
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     *
     * @experimental
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * (experimental) Validate that we don't have any stacks violating dependency order in the pipeline.
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     *
     * @experimental
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    core_1.Annotations.of(this).addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // The path cannot be outside the asm root. I don't really understand how this could ever
        // come to pass, but apparently it has (see https://github.com/aws/aws-cdk/issues/9766).
        // Add a sanity check here so we can catch it more quickly next time.
        if (relativePath.startsWith(`..${path.sep}`)) {
            throw new Error(`The asset manifest (${command.assetManifestPath}) cannot be outside the Cloud Assembly directory (${this.myCxAsmRoot}). Please report this error at https://github.com/aws/aws-cdk/issues to help us debug why this is happening.`);
        }
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
                vpc: this.props.vpc,
                subnetSelection: this.props.subnetSelection,
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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