"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const fs = require("fs");
const path = require("path");
const cfn = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs_1 = require("constructs");
const construct_internals_1 = require("../private/construct-internals");
/**
 * (experimental) Action to deploy a CDK Stack.
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 *
 * @experimental
 */
class DeployCdkStackAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
            templateConfiguration: props.templateConfigurationPath ? props.cloudAssemblyInput.atPath(props.templateConfigurationPath) : undefined,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * (experimental) Construct a DeployCdkStackAction from a Stack artifact.
     *
     * @experimental
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = artifact.environment.account;
        const account = artAccount === core_1.Stack.of(scope).account || artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        const actionRole = roleFromPlaceholderArn(scope, region, account, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, region, account, artifact.cloudFormationExecutionRoleArn);
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        let fullConfigPath;
        if (Object.keys(artifact.tags).length > 0) {
            fullConfigPath = `${fullTemplatePath}.config.json`;
            // Write the template configuration file (for parameters into CreateChangeSet call that
            // cannot be configured any other way). They must come from a file, and there's unfortunately
            // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
            // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
            writeTemplateConfiguration(fullConfigPath, {
                Tags: artifact.tags,
            });
        }
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath: path.relative(appAsmRoot, fullTemplatePath),
            templateConfigurationPath: fullConfigPath ? path.relative(appAsmRoot, fullConfigPath) : undefined,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, region, account, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = constructs_1.Node.of(scope).tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = constructs_1.Node.of(scope).tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
        region: region !== null && region !== void 0 ? region : core_1.Aws.REGION,
        accountId: account !== null && account !== void 0 ? account : core_1.Aws.ACCOUNT_ID,
        partition: core_1.Aws.PARTITION,
    });
    return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false });
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
/**
 * Write template configuration to the given file
 */
function writeTemplateConfiguration(filename, config) {
    fs.writeFileSync(filename, JSON.stringify(config, undefined, 2), { encoding: 'utf-8' });
}
//# sourceMappingURL=data:application/json;base64,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