"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const path = require("path");
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const consts = require("./runtime/consts");
const util_1 = require("./util");
const waiter_state_machine_1 = require("./waiter-state-machine");
const RUNTIME_HANDLER_PATH = path.join(__dirname, 'runtime');
const FRAMEWORK_HANDLER_TIMEOUT = core_1.Duration.minutes(15); // keep it simple for now
/**
 * (experimental) Defines an AWS CloudFormation custom resource provider.
 *
 * @experimental
 */
class Provider extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.isCompleteHandler && (props.queryInterval || props.totalTimeout)) {
            throw new Error('"queryInterval" and "totalTimeout" can only be configured if "isCompleteHandler" is specified. '
                + 'Otherwise, they have no meaning');
        }
        this.onEventHandler = props.onEventHandler;
        this.isCompleteHandler = props.isCompleteHandler;
        this.logRetention = props.logRetention;
        const onEventFunction = this.createFunction(consts.FRAMEWORK_ON_EVENT_HANDLER_NAME);
        if (this.isCompleteHandler) {
            const isCompleteFunction = this.createFunction(consts.FRAMEWORK_IS_COMPLETE_HANDLER_NAME);
            const timeoutFunction = this.createFunction(consts.FRAMEWORK_ON_TIMEOUT_HANDLER_NAME);
            const retry = util_1.calculateRetryPolicy(props);
            const waiterStateMachine = new waiter_state_machine_1.WaiterStateMachine(this, 'waiter-state-machine', {
                isCompleteHandler: isCompleteFunction,
                timeoutHandler: timeoutFunction,
                backoffRate: retry.backoffRate,
                interval: retry.interval,
                maxAttempts: retry.maxAttempts,
            });
            // the on-event entrypoint is going to start the execution of the waiter
            onEventFunction.addEnvironment(consts.WAITER_STATE_MACHINE_ARN_ENV, waiterStateMachine.stateMachineArn);
            waiterStateMachine.grantStartExecution(onEventFunction);
        }
        this.entrypoint = onEventFunction;
        this.serviceToken = this.entrypoint.functionArn;
    }
    /**
     * (deprecated) Called by `CustomResource` which uses this provider.
     *
     * @deprecated use `provider.serviceToken` instead
     */
    bind(_) {
        return {
            serviceToken: this.entrypoint.functionArn,
        };
    }
    createFunction(entrypoint) {
        const fn = new lambda.Function(this, `framework-${entrypoint}`, {
            code: lambda.Code.fromAsset(RUNTIME_HANDLER_PATH),
            description: `AWS CDK resource provider framework - ${entrypoint} (${this.node.path})`.slice(0, 256),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: `framework.${entrypoint}`,
            timeout: FRAMEWORK_HANDLER_TIMEOUT,
            logRetention: this.logRetention,
        });
        fn.addEnvironment(consts.USER_ON_EVENT_FUNCTION_ARN_ENV, this.onEventHandler.functionArn);
        this.onEventHandler.grantInvoke(fn);
        if (this.isCompleteHandler) {
            fn.addEnvironment(consts.USER_IS_COMPLETE_FUNCTION_ARN_ENV, this.isCompleteHandler.functionArn);
            this.isCompleteHandler.grantInvoke(fn);
        }
        return fn;
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,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