"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findLastCommonElement = exports.pathToTopLevelStack = exports.PostResolveToken = exports.filterUndefined = exports.ignoreEmpty = exports.capitalizePropertyNames = void 0;
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value, { stackTrace: false });
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
//# sourceMappingURL=data:application/json;base64,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