"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
// v2 - leave this as a separate section so it reduces merge conflicts when compat is removed
// eslint-disable-next-line import/order
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * (experimental) A construct which represents an AWS resource.
 *
 * @experimental
 */
class Resource extends construct_compat_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        this.env = {
            account: (_a = props.account) !== null && _a !== void 0 ? _a : this.stack.account,
            region: (_b = props.region) !== null && _b !== void 0 ? _b : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    /**
     * @experimental
     */
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * @param arnComponents The format of the ARN of this resource.
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZXNvdXJjZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2RkFBNkY7QUFDN0Ysd0NBQXdDO0FBQ3hDLHlEQUE0RTtBQUc1RSxpQ0FBOEI7QUFDOUIsK0VBQXNHO0FBRXRHLG1DQUFnQztBQUNoQyxtQ0FBZ0M7Ozs7OztBQTBFaEMsTUFBc0IsUUFBUyxTQUFRLDRCQUFhOzs7O0lBa0JoRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXVCLEVBQUU7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsSUFBSSxDQUFDLEtBQUssR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVCLElBQUksQ0FBQyxHQUFHLEdBQUc7WUFDUCxPQUFPLFFBQUUsS0FBSyxDQUFDLE9BQU8sbUNBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO1lBQzVDLE1BQU0sUUFBRSxLQUFLLENBQUMsTUFBTSxtQ0FBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU07U0FDNUMsQ0FBQztRQUNGLElBQUksWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDdEMsSUFBSSxLQUFLLENBQUMsWUFBWSxJQUFJLHFEQUEyQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUN2RSw4Q0FBOEM7WUFDOUMsSUFBSSxDQUFDLGFBQWEsR0FBRyxTQUFTLENBQUM7WUFDL0IsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQztZQUNuQyxZQUFZLEdBQUcsV0FBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQztTQUMxRTthQUNJLElBQUksS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDLGFBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQ3BFLHVDQUF1QztZQUN2QyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7WUFDeEMsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQztTQUN0QzthQUNJO1lBQ0QsdUZBQXVGO1lBQ3ZGLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUN4QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsS0FBSyxDQUFDO1NBQ3ZDO1FBQ0QsSUFBSSxZQUFZLEtBQUssU0FBUyxFQUFFO1lBQzVCLFlBQVksR0FBRyxhQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzVDO1FBQ0QsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7SUFDckMsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSx1QkFBdUI7UUFDMUIsSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRTtZQUM5QixvRUFBb0U7WUFDcEUsTUFBTSxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLG9DQUFvQztnQkFDdEYsNEZBQTRGLENBQUMsQ0FBQztTQUNyRztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3JCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7U0FDcEQ7SUFDTCxDQUFDOzs7O0lBQ1Msb0JBQW9CO1FBQzFCLE9BQU8sOENBQW9CLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQzs7Ozs7Ozs7Ozs7SUFhUyx3QkFBd0IsQ0FBQyxRQUFnQjtRQUMvQyxPQUFPLFdBQUksQ0FBQyxXQUFXLENBQUM7WUFDcEIsT0FBTyxFQUFFLENBQUMsT0FBd0IsRUFBRSxFQUFFO2dCQUNsQyxNQUFNLGNBQWMsR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDL0MsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsS0FBSyxjQUFjLENBQUMsV0FBVyxFQUFFO29CQUN2RCxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztvQkFDL0IsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDO2lCQUM1QjtxQkFDSTtvQkFDRCxPQUFPLFFBQVEsQ0FBQztpQkFDbkI7WUFDTCxDQUFDO1NBQ0osQ0FBQyxDQUFDO0lBQ1AsQ0FBQzs7Ozs7Ozs7Ozs7OztJQW1CUyx1QkFBdUIsQ0FBQyxPQUFlLEVBQUUsYUFBNEI7UUFDM0UsT0FBTyxhQUFLLENBQUMsUUFBUSxDQUFDO1lBQ2xCLE9BQU8sRUFBRSxDQUFDLE9BQXdCLEVBQUUsRUFBRTtnQkFDbEMsTUFBTSxjQUFjLEdBQUcsYUFBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQy9DLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEtBQUssY0FBYyxDQUFDLFdBQVcsRUFBRTtvQkFDdkQsSUFBSSxDQUFDLHVCQUF1QixFQUFFLENBQUM7b0JBQy9CLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUM7aUJBQzlDO3FCQUNJO29CQUNELE9BQU8sT0FBTyxDQUFDO2lCQUNsQjtZQUNMLENBQUM7U0FDSixDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUE5SEQsNEJBOEhDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gdjIgLSBsZWF2ZSB0aGlzIGFzIGEgc2VwYXJhdGUgc2VjdGlvbiBzbyBpdCByZWR1Y2VzIG1lcmdlIGNvbmZsaWN0cyB3aGVuIGNvbXBhdCBpcyByZW1vdmVkXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBJQ29uc3RydWN0LCBDb25zdHJ1Y3QgYXMgQ29yZUNvbnN0cnVjdCB9IGZyb20gJy4vY29uc3RydWN0LWNvbXBhdCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEFybkNvbXBvbmVudHMgfSBmcm9tICcuL2Fybic7XG5pbXBvcnQgeyBMYXp5IH0gZnJvbSAnLi9sYXp5JztcbmltcG9ydCB7IGdlbmVyYXRlUGh5c2ljYWxOYW1lLCBpc0dlbmVyYXRlZFdoZW5OZWVkZWRNYXJrZXIgfSBmcm9tICcuL3ByaXZhdGUvcGh5c2ljYWwtbmFtZS1nZW5lcmF0b3InO1xuaW1wb3J0IHsgSVJlc29sdmVDb250ZXh0IH0gZnJvbSAnLi9yZXNvbHZhYmxlJztcbmltcG9ydCB7IFN0YWNrIH0gZnJvbSAnLi9zdGFjayc7XG5pbXBvcnQgeyBUb2tlbiB9IGZyb20gJy4vdG9rZW4nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlRW52aXJvbm1lbnQge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGFjY291bnQ6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcmVnaW9uOiBzdHJpbmc7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlc291cmNlIGV4dGVuZHMgSUNvbnN0cnVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc3RhY2s6IFN0YWNrO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZW52OiBSZXNvdXJjZUVudmlyb25tZW50O1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHBoeXNpY2FsTmFtZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYWNjb3VudD86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgUmVzb3VyY2UgZXh0ZW5kcyBDb3JlQ29uc3RydWN0IGltcGxlbWVudHMgSVJlc291cmNlIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgc3RhY2s6IFN0YWNrO1xuICAgIHB1YmxpYyByZWFkb25seSBlbnY6IFJlc291cmNlRW52aXJvbm1lbnQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IHBoeXNpY2FsTmFtZTogc3RyaW5nO1xuICAgIHByaXZhdGUgX3BoeXNpY2FsTmFtZTogc3RyaW5nIHwgdW5kZWZpbmVkO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX2FsbG93Q3Jvc3NFbnZpcm9ubWVudDogYm9vbGVhbjtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVzb3VyY2VQcm9wcyA9IHt9KSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAgICAgdGhpcy5lbnYgPSB7XG4gICAgICAgICAgICBhY2NvdW50OiBwcm9wcy5hY2NvdW50ID8/IHRoaXMuc3RhY2suYWNjb3VudCxcbiAgICAgICAgICAgIHJlZ2lvbjogcHJvcHMucmVnaW9uID8/IHRoaXMuc3RhY2sucmVnaW9uLFxuICAgICAgICB9O1xuICAgICAgICBsZXQgcGh5c2ljYWxOYW1lID0gcHJvcHMucGh5c2ljYWxOYW1lO1xuICAgICAgICBpZiAocHJvcHMucGh5c2ljYWxOYW1lICYmIGlzR2VuZXJhdGVkV2hlbk5lZWRlZE1hcmtlcihwcm9wcy5waHlzaWNhbE5hbWUpKSB7XG4gICAgICAgICAgICAvLyBhdXRvLWdlbmVyYXRlIG9ubHkgaWYgY3Jvc3MtZW52IGlzIHJlcXVpcmVkXG4gICAgICAgICAgICB0aGlzLl9waHlzaWNhbE5hbWUgPSB1bmRlZmluZWQ7XG4gICAgICAgICAgICB0aGlzLl9hbGxvd0Nyb3NzRW52aXJvbm1lbnQgPSB0cnVlO1xuICAgICAgICAgICAgcGh5c2ljYWxOYW1lID0gTGF6eS5zdHJpbmdWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX3BoeXNpY2FsTmFtZSB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmIChwcm9wcy5waHlzaWNhbE5hbWUgJiYgIVRva2VuLmlzVW5yZXNvbHZlZChwcm9wcy5waHlzaWNhbE5hbWUpKSB7XG4gICAgICAgICAgICAvLyBjb25jcmV0ZSB2YWx1ZSBzcGVjaWZpZWQgYnkgdGhlIHVzZXJcbiAgICAgICAgICAgIHRoaXMuX3BoeXNpY2FsTmFtZSA9IHByb3BzLnBoeXNpY2FsTmFtZTtcbiAgICAgICAgICAgIHRoaXMuX2FsbG93Q3Jvc3NFbnZpcm9ubWVudCA9IHRydWU7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAvLyBlaXRoZXIgdW5kZWZpbmVkIChkZXBsb3ktdGltZSkgb3IgaGFzIHRva2Vucywgd2hpY2ggbWVhbnMgd2UgY2FuJ3QgdXNlIGZvciBjcm9zcy1lbnZcbiAgICAgICAgICAgIHRoaXMuX3BoeXNpY2FsTmFtZSA9IHByb3BzLnBoeXNpY2FsTmFtZTtcbiAgICAgICAgICAgIHRoaXMuX2FsbG93Q3Jvc3NFbnZpcm9ubWVudCA9IGZhbHNlO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwaHlzaWNhbE5hbWUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgcGh5c2ljYWxOYW1lID0gVG9rZW4uYXNTdHJpbmcodW5kZWZpbmVkKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnBoeXNpY2FsTmFtZSA9IHBoeXNpY2FsTmFtZTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2FsbGVkIHdoZW4gdGhpcyByZXNvdXJjZSBpcyByZWZlcmVuY2VkIGFjcm9zcyBlbnZpcm9ubWVudHNcbiAgICAgKiAoYWNjb3VudC9yZWdpb24pIHRvIG9yZGVyIHRvIHJlcXVlc3QgdGhhdCBhIHBoeXNpY2FsIG5hbWUgd2lsbCBiZSBnZW5lcmF0ZWRcbiAgICAgKiBmb3IgdGhpcyByZXNvdXJjZSBkdXJpbmcgc3ludGhlc2lzLCBzbyB0aGUgcmVzb3VyY2UgY2FuIGJlIHJlZmVyZW5jZWRcbiAgICAgKiB0aHJvdWdoIGl0J3MgYWJzb2x1dGUgbmFtZS9hcm4uXG4gICAgICpcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgX2VuYWJsZUNyb3NzRW52aXJvbm1lbnQoKTogdm9pZCB7XG4gICAgICAgIGlmICghdGhpcy5fYWxsb3dDcm9zc0Vudmlyb25tZW50KSB7XG4gICAgICAgICAgICAvLyBlcnJvciBvdXQgLSBhIGRlcGxveS10aW1lIG5hbWUgY2Fubm90IGJlIHVzZWQgYWNyb3NzIGVudmlyb25tZW50c1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgdXNlIHJlc291cmNlICcke3RoaXMubm9kZS5wYXRofScgaW4gYSBjcm9zcy1lbnZpcm9ubWVudCBmYXNoaW9uLCBgICtcbiAgICAgICAgICAgICAgICBcInRoZSByZXNvdXJjZSdzIHBoeXNpY2FsIG5hbWUgbXVzdCBiZSBleHBsaWNpdCBzZXQgb3IgdXNlIGBQaHlzaWNhbE5hbWUuR0VORVJBVEVfSUZfTkVFREVEYFwiKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoIXRoaXMuX3BoeXNpY2FsTmFtZSkge1xuICAgICAgICAgICAgdGhpcy5fcGh5c2ljYWxOYW1lID0gdGhpcy5nZW5lcmF0ZVBoeXNpY2FsTmFtZSgpO1xuICAgICAgICB9XG4gICAgfVxuICAgIHByb3RlY3RlZCBnZW5lcmF0ZVBoeXNpY2FsTmFtZSgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gZ2VuZXJhdGVQaHlzaWNhbE5hbWUodGhpcyk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHJvdGVjdGVkIGdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShuYW1lQXR0cjogc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiBMYXp5LnN0cmluZ1ZhbHVlKHtcbiAgICAgICAgICAgIHByb2R1Y2U6IChjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpID0+IHtcbiAgICAgICAgICAgICAgICBjb25zdCBjb25zdW1pbmdTdGFjayA9IFN0YWNrLm9mKGNvbnRleHQuc2NvcGUpO1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLnN0YWNrLmVudmlyb25tZW50ICE9PSBjb25zdW1pbmdTdGFjay5lbnZpcm9ubWVudCkge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl9lbmFibGVDcm9zc0Vudmlyb25tZW50KCk7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiB0aGlzLnBoeXNpY2FsTmFtZTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBuYW1lQXR0cjtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCBnZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShhcm5BdHRyOiBzdHJpbmcsIGFybkNvbXBvbmVudHM6IEFybkNvbXBvbmVudHMpIHtcbiAgICAgICAgcmV0dXJuIFRva2VuLmFzU3RyaW5nKHtcbiAgICAgICAgICAgIHJlc29sdmU6IChjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpID0+IHtcbiAgICAgICAgICAgICAgICBjb25zdCBjb25zdW1pbmdTdGFjayA9IFN0YWNrLm9mKGNvbnRleHQuc2NvcGUpO1xuICAgICAgICAgICAgICAgIGlmICh0aGlzLnN0YWNrLmVudmlyb25tZW50ICE9PSBjb25zdW1pbmdTdGFjay5lbnZpcm9ubWVudCkge1xuICAgICAgICAgICAgICAgICAgICB0aGlzLl9lbmFibGVDcm9zc0Vudmlyb25tZW50KCk7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiB0aGlzLnN0YWNrLmZvcm1hdEFybihhcm5Db21wb25lbnRzKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBhcm5BdHRyO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbiJdfQ==