"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.synthesize = void 0;
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const annotations_1 = require("../annotations");
const aspect_1 = require("../aspect");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const metadata_resource_1 = require("./metadata-resource");
const prepare_app_1 = require("./prepare-app");
const tree_metadata_1 = require("./tree-metadata");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    injectMetadataResources(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly();
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...aspects.aspects];
        const nodeAspectsCount = aspects.aspects.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.aspects.length) {
                annotations_1.Annotations.of(construct).addWarning('We detected an Aspect was added via another Aspect, and will not be applied');
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Find all stacks and add Metadata Resources to all of them
 *
 * There is no good generic place to do this. Can't do it in the constructor
 * (because adding a child construct makes it impossible to set context on the
 * node), and the generic prepare phase is deprecated.
 *
 * Only do this on [parent] stacks (not nested stacks), don't do this when
 * disabled by the user.
 *
 * Also, only when running via the CLI. If we do it unconditionally,
 * all unit tests everywhere are going to break massively. I've spent a day
 * fixing our own, but downstream users would be affected just as badly.
 *
 * Stop at Assembly boundaries.
 */
function injectMetadataResources(root) {
    visit(root, 'post', construct => {
        if (!stack_1.Stack.isStack(construct) || !construct._versionReportingEnabled) {
            return;
        }
        // Because of https://github.com/aws/aws-cdk/blob/master/packages/@aws-cdk/assert/lib/synth-utils.ts#L74
        // synthesize() may be called more than once on a stack in unit tests, and the below would break
        // if we execute it a second time. Guard against the constructs already existing.
        const CDKMetadata = 'CDKMetadata';
        if (construct.node.tryFindChild(CDKMetadata)) {
            return;
        }
        new metadata_resource_1.MetadataResource(construct, CDKMetadata);
    });
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => {
        const session = {
            outdir: builder.outdir,
            assembly: builder,
        };
        if (construct instanceof stack_1.Stack) {
            construct.synthesizer.synthesize(session);
        }
        else if (construct instanceof tree_metadata_1.TreeMetadata) {
            construct._synthesizeTree(session);
        }
        // this will soon be deprecated and removed in 2.x
        // see https://github.com/aws/aws-cdk-rfcs/issues/192
        construct.onSynthesize(session);
    });
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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