"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs', '@amzn'];
// list of NPM packages included in version reporting
const WHITELIST_PACKAGES = ['aws-rfdk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || WHITELIST_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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