"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetadataResource = void 0;
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const region_info_1 = require("../../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const cfn_condition_1 = require("../cfn-condition");
const cfn_fn_1 = require("../cfn-fn");
const cfn_pseudo_1 = require("../cfn-pseudo");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const lazy_1 = require("../lazy");
const token_1 = require("../token");
const runtime_info_1 = require("./runtime-info");
/**
 * Construct that will render the metadata resource
 */
class MetadataResource extends construct_compat_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const metadataServiceExists = token_1.Token.isUnresolved(scope.region) || region_info_1.RegionInfo.get(scope.region).cdkMetadataResourceAvailable;
        if (metadataServiceExists) {
            const resource = new cfn_resource_1.CfnResource(this, 'Default', {
                type: 'AWS::CDK::Metadata',
                properties: {
                    Modules: lazy_1.Lazy.stringValue({ produce: () => MetadataResource.modulesProperty() }),
                },
            });
            // In case we don't actually know the region, add a condition to determine it at deploy time
            if (token_1.Token.isUnresolved(scope.region)) {
                const condition = new cfn_condition_1.CfnCondition(this, 'Condition', {
                    expression: makeCdkMetadataAvailableCondition(),
                });
                // To not cause undue template changes
                condition.overrideLogicalId('CDKMetadataAvailable');
                resource.cfnOptions.condition = condition;
            }
        }
    }
    /**
     * Clear the modules cache
     *
     * The next time the MetadataResource is rendered, it will do a lookup of the
     * modules from the NodeJS module cache again.
     *
     * Used only for unit tests.
     */
    static clearModulesCache() {
        this._modulesPropertyCache = undefined;
    }
    /**
     * Calculate the modules property
     */
    static modulesProperty() {
        if (this._modulesPropertyCache === undefined) {
            this._modulesPropertyCache = formatModules(runtime_info_1.collectRuntimeInformation());
        }
        return this._modulesPropertyCache;
    }
}
exports.MetadataResource = MetadataResource;
function makeCdkMetadataAvailableCondition() {
    return cfn_fn_1.Fn.conditionOr(...region_info_1.RegionInfo.regions
        .filter(ri => ri.cdkMetadataResourceAvailable)
        .map(ri => cfn_fn_1.Fn.conditionEquals(cfn_pseudo_1.Aws.REGION, ri.name)));
}
function formatModules(runtime) {
    const modules = new Array();
    // inject toolkit version to list of modules
    const cliVersion = process.env[cxapi.CLI_VERSION_ENV];
    if (cliVersion) {
        modules.push(`aws-cdk=${cliVersion}`);
    }
    for (const key of Object.keys(runtime.libraries).sort()) {
        modules.push(`${key}=${runtime.libraries[key]}`);
    }
    return modules.join(',');
}
//# sourceMappingURL=data:application/json;base64,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