"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResource = void 0;
const cfn_resource_1 = require("./cfn-resource");
const removal_policy_1 = require("./removal-policy");
const resource_1 = require("./resource");
const token_1 = require("./token");
/**
 * (experimental) Custom resource that is implemented using a Lambda.
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 *
 * @experimental
 * @resource AWS::CloudFormation::CustomResource
 */
class CustomResource extends resource_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const pascalCaseProperties = (_a = props.pascalCaseProperties) !== null && _a !== void 0 ? _a : false;
        const properties = pascalCaseProperties ? uppercaseProperties(props.properties || {}) : (props.properties || {});
        this.resource = new cfn_resource_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.serviceToken,
                ...properties,
            },
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: removal_policy_1.RemovalPolicy.DESTROY,
        });
    }
    /**
     * (experimental) The physical name of this custom resource.
     *
     * @experimental
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * (experimental) Returns the value of an attribute of the custom resource of an arbitrary type.
     *
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute.
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     * @experimental
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * (experimental) Returns the value of an attribute of the custom resource of type string.
     *
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute.
     * @returns a token for `Fn::GetAtt` encoded as a string.
     * @experimental
     */
    getAttString(attributeName) {
        return token_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return 'AWS::CloudFormation::CustomResource';
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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