"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const fs = require("fs");
const path = require("path");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * (experimental) An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const asset = stack.addFileAsset({
            fileName: staging.stagedPath,
            sourceHash: staging.sourceHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        const timeout = (_a = props.timeout) !== null && _a !== void 0 ? _a : duration_1.Duration.minutes(15);
        const memory = (_b = props.memorySize) !== null && _b !== void 0 ? _b : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: 'nodejs12.x',
            },
        });
        handler.addDependsOn(role);
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * (experimental) Returns a stack-level singleton ARN (service token) for the custom resource provider.
     *
     * @param scope Construct scope.
     * @param uniqueid A globally unique id that will be used for the stack-level construct.
     * @param props Provider properties which will only be applied when the provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     * @experimental
     */
    static getOrCreate(scope, uniqueid, props) {
        var _a;
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_a = stack.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new CustomResourceProvider(stack, id, props);
        return provider.serviceToken;
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,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