"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * (experimental) Construct to import an existing CloudFormation template file into a CDK application.
 *
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 *
 * @experimental
 */
class CfnInclude extends core.CfnElement {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * (experimental) Returns the low-level CfnResource from the template with the given logical ID.
     *
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *      // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *      const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *      // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file.
     * @experimental
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnCondition object from the 'Conditions' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file.
     * @experimental
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnParameter object from the 'Parameters' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve.
     * @experimental
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnMapping object from the 'Mappings' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve.
     * @experimental
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnOutput object from the 'Outputs' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve.
     * @experimental
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnRule object from the 'Rules' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template.
     * @experimental
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnHook object from the 'Hooks' section of the included CloudFormation template with the given logical ID.
     *
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section.
     * @experimental
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns a loaded NestedStack with name logicalId.
     *
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @experimental
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * (experimental) Includes a template for a child stack inside of this parent template.
     *
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @param nestedStackProps the properties of the included child Stack.
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     * @experimental
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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