"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegrationPattern = exports.TaskStateBase = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const chain_1 = require("../chain");
const state_1 = require("./state");
/**
 * (experimental) Define a Task state in the state machine.
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 *
 * @experimental
 */
class TaskStateBase extends state_1.State {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.endStates = [this];
        this.timeout = props.timeout;
        this.heartbeat = props.heartbeat;
    }
    /**
     * (experimental) Add retry configuration for this state.
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     *
     * @experimental
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * (experimental) Add a recovery handler for this state.
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     *
     * @experimental
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * (experimental) Continue normal execution with the given state.
     *
     * @experimental
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * (experimental) Return the Amazon States Language object for this state.
     *
     * @experimental
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderTaskBase(),
            ...this._renderTask(),
        };
    }
    /**
     * (experimental) Return the given named metric for this Task.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metric(metricName, props) {
        var _a;
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: (_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricDimensions,
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default - average over 5 minutes
     * @experimental
     */
    metricRunTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * (experimental) The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default - average over 5 minutes
     * @experimental
     */
    metricScheduleTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * (experimental) The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default - average over 5 minutes
     * @experimental
     */
    metricTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * (experimental) Metric for the number of times this activity is scheduled.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricScheduled(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * (experimental) Metric for the number of times this activity times out.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricTimedOut(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * (experimental) Metric for the number of times this activity is started.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricStarted(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Started', props);
    }
    /**
     * (experimental) Metric for the number of times this activity succeeds.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricSucceeded(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * (experimental) Metric for the number of times this activity fails.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricFailed(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Failed', props);
    }
    /**
     * (experimental) Metric for the number of times the heartbeat times out for this activity.
     *
     * @default - sum over 5 minutes
     * @experimental
     */
    metricHeartbeatTimedOut(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    /**
     * (experimental) Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @experimental
     */
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskPolicies || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        }
        return this.metric(prefix + suffix, props);
    }
    renderTaskBase() {
        var _a, _b;
        return {
            Type: 'Task',
            Comment: this.comment,
            TimeoutSeconds: (_a = this.timeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            HeartbeatSeconds: (_b = this.heartbeat) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            InputPath: state_1.renderJsonPath(this.inputPath),
            OutputPath: state_1.renderJsonPath(this.outputPath),
            ResultPath: state_1.renderJsonPath(this.resultPath),
        };
    }
}
exports.TaskStateBase = TaskStateBase;
/**
 * (experimental) AWS Step Functions integrates with services directly in the Amazon States Language.
 *
 * You can control these AWS services using service integration patterns:
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 * @experimental
 */
var IntegrationPattern;
(function (IntegrationPattern) {
    IntegrationPattern["REQUEST_RESPONSE"] = "REQUEST_RESPONSE";
    IntegrationPattern["RUN_JOB"] = "RUN_JOB";
    IntegrationPattern["WAIT_FOR_TASK_TOKEN"] = "WAIT_FOR_TASK_TOKEN";
})(IntegrationPattern = exports.IntegrationPattern || (exports.IntegrationPattern = {}));
//# sourceMappingURL=data:application/json;base64,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