"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
/**
 * (experimental) A collection of connected states.
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *    be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *    machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 *
 * @experimental
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched.
     * @param graphDescription description of the state machine.
     * @experimental
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * (experimental) The accumulated policy statements.
         *
         * @experimental
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * (experimental) Register a state as part of this graph.
     *
     * Called by State.bindToGraph().
     *
     * @experimental
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * (experimental) Register a Policy Statement used by states in this graph.
     *
     * @experimental
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * (experimental) Register this graph as a child of the given graph.
     *
     * Resource changes will be bubbled up to the given graph.
     *
     * @experimental
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * (experimental) Return the Amazon States Language JSON for this graph.
     *
     * @experimental
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * (experimental) Return a string description of this graph.
     *
     * @experimental
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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