"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionsStartExecution = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to call StartExecution on another state machine.
 *
 * It supports three service integration patterns: FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.
 *
 * @experimental
 */
class StepFunctionsStartExecution extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, StepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
     * @internal
     */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        return {
            Resource: `${task_utils_1.integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: this.props.input ? this.props.input.value : sfn.TaskInput.fromDataAt('$').value,
                StateMachineArn: this.props.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
     * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
     *
     * the scoped access policy should be generated accordingly.
     *
     * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
     */
    createScopedAccessPolicy() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.props.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['states:DescribeExecution', 'states:StopExecution'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        sep: ':',
                        resourceName: `${stack.parseArn(this.props.stateMachine.stateMachineArn, ':').resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.StepFunctionsStartExecution = StepFunctionsStartExecution;
StepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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