"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTransformJob = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * (experimental) Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SageMakerCreateTransformJob extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = props.transformInput.transformDataSource.s3DataSource.s3DataType
            ? props.transformInput
            : Object.assign({}, props.transformInput, {
                transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    /**
     * (experimental) The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     *
     * @experimental
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy ? { BatchStrategy: this.props.batchStrategy } : {}),
            ...this.renderEnvironment(this.props.environment),
            ...(this.props.maxConcurrentTransforms ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {}),
            ...(this.props.maxPayload ? { MaxPayloadInMB: this.props.maxPayload.toMebibytes() } : {}),
            ModelName: this.props.modelName,
            ...utils_1.renderTags(this.props.tags),
            ...this.renderTransformInput(this.transformInput),
            TransformJobName: this.props.transformJobName,
            ...this.renderTransformOutput(this.props.transformOutput),
            ...this.renderTransformResources(this.transformResources),
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType ? { CompressionType: input.compressionType } : {}),
                ...(input.contentType ? { ContentType: input.contentType } : {}),
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType ? { SplitType: input.splitType } : {}),
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey ? { KmsKeyId: output.encryptionKey.keyArn } : {}),
                ...(output.accept ? { Accept: output.accept } : {}),
                ...(output.assembleWith ? { AssembleWith: output.assembleWith } : {}),
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: 'ml.' + resources.instanceType,
                ...(resources.volumeEncryptionKey ? { VolumeKmsKeyId: resources.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderEnvironment(environment) {
        return environment ? { Environment: environment } : {};
    }
    makePolicyStatements() {
        const stack = core_1.Stack.of(this);
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(this, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')],
            });
        }
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTransformJob = SageMakerCreateTransformJob;
SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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