"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Saves the received message to an Amazon S3 bucket and, optionally, publishes a notification to Amazon SNS.
 *
 * @experimental
 */
class S3 {
    /**
     * @experimental
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * (experimental) Returns the receipt rule action specification.
     *
     * @experimental
     */
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            cdk.Annotations.of(rule).addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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